/*
 * 915 MHz Wireless Remote Coax Switch - Remote unit
 * 
 * by Glen Popiel - KW5GP
 * 
 * uses Adafruit Feather 32U4 LoRa
 * 
 *  uses Radiohead library by Mike McCauley, mikem@airspayce.com
 */

#define debug_mode 1

#include <SPI.h>  // Built-in SPI library
#include <RH_RF95.h>  // Radiohead RF95 library

// Define the Feather32U4 pins
#define RFM95_CS 8
#define RFM95_RST 4
#define RFM95_INT 7

// Define the radio frequency - must match remote's freq!
#define RF95_FREQ 915.0

uint8_t buf[RH_RF95_MAX_MESSAGE_LEN]; // Receive data buffer - set to RF95 Maximum Message Length
uint8_t len = sizeof(buf);  // Variable to hold the size of the Rx buffer

// Create an instance of the radio driver
RH_RF95 rf95(RFM95_CS, RFM95_INT);

// Define the onboard LED
#define LED 13

// Load the pin numbers for the relays and LED into an array
int relay[5] = {0, 5, 6, 10, 11};
int led[5] = {0,18,19,20,21};

// Variables to hold the value of the current relay and the relay needing to be set
int current_relay = 0, relay_to_set = 0;

// Variable to hold data to be send back
uint8_t data[3];

void setup()
{
  if (debug_mode)
  {
    while (!Serial);  
    Serial.begin(9600);
    delay(100);
    Serial.println("Starting");
  }
  
  // Set the onboard LED pin mode
  pinMode(LED, OUTPUT);

  // Set the LED and relay pin modes
  for (int x=1; x<=5; x++)
  {
    pinMode(relay[x], OUTPUT);
    pinMode(led[x], OUTPUT);
  }

  // set the radio reset pinmode and turn off radio reset
  pinMode(RFM95_RST, OUTPUT);
  digitalWrite(RFM95_RST, HIGH);

  if (debug_mode)
  {
    Serial.println("Feather LoRa RX Test!");
  }
  
  // reset the radio
  digitalWrite(RFM95_RST, LOW);
  delay(100);
  digitalWrite(RFM95_RST, HIGH);
  delay(100);

  if (debug_mode)
  {
    Serial.println("Initializing Radio");
  }
  
  // Initialize the RF95 radio
  // Defaults after init are 434.0MHz, modulation GFSK_Rb250Fd250, +13dBm, Bw = 125 kHz, Cr = 4/5, Sf = 128chips/symbol, CRC on
  while (!rf95.init()) 
  {
    if (debug_mode)
    {
      Serial.println("LoRa radio init failed");
    }
    while (1);
  }

  if (debug_mode)
  {
    Serial.println("LoRa radio init OK!");
  }

  // Set the radio frequency
  if (!rf95.setFrequency(RF95_FREQ)) 
  {
    if (debug_mode)
    {
      Serial.println("setFrequency failed");
    }
    while (1);
  }

  if (debug_mode)
  {
    Serial.print("Set Freq to: "); Serial.println(RF95_FREQ);
  }

  // The default transmitter power is 13dBm, using PA_BOOST.
  // If you are using RFM95/96/97/98 modules which uses the PA_BOOST transmitter pin, then
  // you can set transmitter powers from 5 to 23 dBm:
  
  // Set the TX power to 23dBm
  rf95.setTxPower(23, false);
  
}

void loop()

{
  // Check to see if radio is available
  if (rf95.available())
  {
    // Check to see if a message has been received
    if (rf95.recv(buf, &len))
    {
      // Turn on the onboard LED to indicate a message received
      digitalWrite(LED, HIGH);
  
      RH_RF95::printBuffer("Received: ", buf, len);
      if (debug_mode)
      {    
        Serial.print("Max Length :");
        Serial.println(RH_RF95_MAX_MESSAGE_LEN);
        Serial.print("Got: ");
        Serial.println((char*)buf);
      }
      // Save the receive data to a String
      String test = ((char*)buf);
      if (debug_mode)
      {
        Serial.println(test);
        Serial.print("RSSI: ");
        Serial.println(rf95.lastRssi(), DEC);
      }

      // Set the default response to "XX" to indicate no valid data received
      uint8_t data[] = "XX";
      
      // Check for Set Relay command code ('S')
      if (test[0] == 'S')
      {
        // Set relay command
        if (debug_mode)
        {
          Serial.println("Set Relay Command Received");
        }
        // The second character in the command code is the relay number to set
        int relay_to_set = test[1];
        // Convert from ASCII value to numeric value
        relay_to_set = relay_to_set - 48;

        if (debug_mode)
        {
          Serial.print("test[1] = ");
          Serial.println(test[1]);
        }
        
        // Call the set_Relay function
        set_Relay(relay_to_set);
        
        // Set the response code to 'C' to indicate command confirmed
        data[0] = int('C');
      }

      // Check for Read Relay command code ('R')
      if (test[0] == 'R')
      {
        // Return Relay Info command

        if (debug_mode)
        {
          Serial.println("Return Relay Info Command Received");
        }
        
        // Set the response code to 'R' to indicate return current relay command
        data[0] = int('R');        
      }

      // Send a reply back if we received a valid command code
      if (test[0] == 83 || test[0] == 82)
      {
        // Send a reply
        // Add a zero at the end of the data array to indicate end of message data
        data[2] = 0;
        
        // Set the second character to ASCII value of relay set
        data[1] = int('0' + current_relay);        
        
        if (debug_mode)
        {
          Serial.print("Data to Send : ");
          Serial.print(char(data[0]));
          Serial.print("   ");
          Serial.println(char(data[1]));
        }
        
        // Send the data and wait for confimation it was sent
        rf95.send(data, sizeof(data));
        rf95.waitPacketSent();
        
        if (debug_mode)
        {
          Serial.println("Sent a reply");
        }

        // Turn off the onboard LED to indicate processing is complete
        digitalWrite(LED, LOW);
      }
    }
    else
    {
      if (debug_mode)
      {
        Serial.println("Receive failed");
      }
    }
  }
  
}

// Function to set the desired relay
void set_Relay(int relay_number)
{
  // Turn off the last relay
  if (current_relay != 0)
  {
    digitalWrite(relay[current_relay], LOW);
    digitalWrite(led[current_relay], LOW);
  }
  if (relay_number != 0)
  {
    // Turn off the previous relay if it was set
    digitalWrite(relay[current_relay], LOW);
    digitalWrite(led[current_relay], LOW);
    
    // Set the appropriate relay
    if (debug_mode)
    {
      Serial.print("Relay to function = ");
      Serial.println(relay_number);
    }

    
    digitalWrite(relay[relay_number], HIGH);
    digitalWrite(led[relay_number], HIGH);
  }

  // Set the current relay to the relay we just set
  current_relay = relay_number;

}



