/*
40m QRP Transceiver with DDS VFO, Iambic Keyer, Dual Speed RIT, Dual Speed Tuning 
December 2014
Glen Popiel - KW5GP

Uses M0XPD DDS Library

Uses OpenQRP Blog Iambic Morse Code Keyer Sketch by Steven T. Elliott, K1EL - Used with Permission

// Modified by Glen Popiel, KW5GP

/////////////////////////////////////////////////////////////////////////////////////////
//
//  Includes Iambic Morse Code Keyer Sketch
//  Copyright (c) 2009 Steven T. Elliott
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License, or (at your option) any later version.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details:
//
//  Free Software Foundation, Inc., 59 Temple Place, Suite 330,
//  Boston, MA  02111-1307  USA
//
/////////////////////////////////////////////////////////////////////////////////////////

*/

// include the M0XPD DDS Library:
#include <DDS.h>
// Include the Nokia 5110 Library
#include <LCD5110_Basic.h>
// Include the Rotary Encoder Library
#include <Encoder.h>

//======================================
// AD9850 Module....
// set pin numbers:
const int W_CLK = 13;
const int FQ_UD = 10; 
const int DATA = 11;
const int RESET = 9;

// Nokia 5110 LCD Module
// Set Pin Numbers
const int CLK = 8;
const int DIN = 7;
const int DC = 6;
const int RST = 4;
const int CE = 5;

const int rit_pot = 0;  // Set the RIT Pot pin to A0
const int rit_fast_step = 5000;  // The RIT fast step rate
const int rit_slow_step = 500;  // The RIT slow step rate
const int encoder_PinA = 2;  // Pin A of the Rotary Encoder
const int encoder_PinB = 3;  // Pin B of the Rotary Encoder
const int encoder_Switch_Pin = A4;  // Encoder Pushbutton Switch input pin
const int speed_Pot = A1;  // CW Speed Pot Pin
const int key_dit = A3;  // Keyer Dit input pin
const int key_dah = A2;  // Keyer Dah input pin
const int key_pin = 12;  // Key Out pin
const int key_mode = 1; // Set to 0 for Iambic A mode. Set to 1 for Iambic B, Set to 2 for Straight Key
const int debounceInterval = 500;  //disable step change for 500 ms between encoder pushbutton press
const int fast_step = 100;  // Rotary Encoder Fast Tuning Step
const int slow_step = 1;  // Rotary Encoder Slow Tuning Step
const int battery_pin = A5; // Battery Voltage Input pin

const double band_low = 7000000; // Low end of Band
const double band_high = 7125000; // High end of CW portion of band

int tx_offset = 600;
int rit;
int old_rit = 0;
int rit_freq;
int rit_range;
int encoder_Pos = 0;  
int old_encoder_Pos = 0;
int cw_speed = 5;
int old_speed = 0;

int step_size = fast_step;

double freq = 7030000;  // Starting freq
double tx_freq = freq;  // set the TX freq to the starting freq
double rx_freq = freq;  // set the RX freq to the starting freq
double dds_freq = rx_freq;

boolean encoder_Switch;  
boolean current_step = HIGH;  // Default Tuning Step is Fast

unsigned long int delay_time = 0;

float battery_volts;
float volt_drop = 0.89;
int volt_cal = 2490;

// Instantiate the DDS
DDS dds(W_CLK, FQ_UD, DATA, RESET);

  
// Instantiate the LCD
LCD5110 glcd(CLK,DIN,DC,RST,CE); //  Assign the Nokia 5110 LCD Pins

// Instantiate the Rotary Encoder
Encoder Enc(encoder_PinA, encoder_PinB);

extern uint8_t SmallFont[];  // define the Nokia Font  

// Keyer Variables
unsigned long ditTime;  // Number of milliseconds per dit
char keyerControl;
char keyerState;
static long ktimer;

/////////////////////////////////////////////////////////////////////////////////////////
//  keyerControl bit definitions
//
#define     DIT_L      0x01     // Dit latch
#define     DAH_L      0x02     // Dah latch
#define     DIT_PROC   0x04     // Dit is being processed
#define     PDLSWAP    0x08     // 0 for normal, 1 for swap
#define     IAMBICB    0x10     // 0 for Iambic A, 1 for Iambic B
#define     ULTIMATIC  0x20     // 1 for ultimatic
#define     STRAIGHT   0x80     // 1 for straight key mode 
////////////////////////////////////////////////////////////////////////////////////////
//
//  State Machine Defines

enum KSTYPE {IDLE, CHK_DIT, CHK_DAH, KEYED_PREP, KEYED, INTER_ELEMENT };

void setup() 
{
  
  // Set up the Nokia 5110 Display
  glcd.InitLCD(65);  // Initialize 
  glcd.setFont(SmallFont);  // Use Small Font
  
  // Display the Startup screen
  glcd.clrScr();
  glcd.print("KW5GP", CENTER,0);
  glcd.print("40m QRP",CENTER,8);
  glcd.print("Transceiver",CENTER,16);
  glcd.print("Initializing", CENTER,40);
  
  // start up the DDS...   
  dds.init();  
  // (Optional) trim if your xtal is not at 125MHz...
  dds.trim(125000322); // enter actual osc freq 

  
  // Set up the Rotary Encoder
  pinMode (encoder_PinA,INPUT);
  pinMode (encoder_PinB,INPUT);
  pinMode (encoder_Switch_Pin, INPUT);
  // Enable the Internal Pull-up resistor on the Encoder Inputs  
  digitalWrite(encoder_PinA, HIGH);
  digitalWrite(encoder_PinB, HIGH);
  digitalWrite(encoder_Switch_Pin, HIGH);
  
  // Set up the Keyer
  pinMode (key_dit, INPUT);
  pinMode (key_dah, INPUT);
  pinMode (key_pin, OUTPUT);
  // Enable the Internal Pull-up resistor on the Keyer Inputs  
  digitalWrite(key_dit, HIGH);
  digitalWrite(key_dah, HIGH);

  // Setup the Keyer  
  keyerState = IDLE;
  keyerControl = 0;

  
  // Setup the Battery Voltage input
  pinMode (battery_pin, INPUT);
  
  delay(5000);
  glcd.clrScr();
  
  // Setup the LCD Display for operation
  glcd.print("TX:",0,0);  // Display the Frequency information on the LCD
  glcd.print("RX:",0,8);
  glcd.print("Step:",0,24);
  glcd.print("Batt:",0,40);
  
 // glcd.print("DDS:",0,16);  // Display the DDS Frequency
  
  // start the oscillator...
  set_DDS(rx_freq);
  
  if (key_mode != 2)
  {
    glcd.print("Speed:",0,32);
    glcd.print("wpm",55,32); 
    glcd.printNumI(cw_speed,40,32,2); // Update the CW Speed display    
  } else {
    glcd.print("Straight Key",0,32);
  }
  
  // print the current RIT and encoder position
  glcd.print("Fast",35,24);

  glcd.printNumI(tx_freq,30,0,7);  // display the Tx and Rx Freq
  glcd.printNumI(rx_freq,30,8,7);
//  glcd.printNumI(dds_freq,30,16,7);  // Display the DDS Frequency
  
}  // End Setup Loop

void loop() 
{
  if (keyerState != KEYED)
  {
    read_rit();  // Read and set the RIT
  
    set_keyer_speed(); // Read and set the Keyer Speed
  
    read_encoder(); // Read the encoder and set the Tx/Rx frequencies

    read_pushbutton();  // Read the Encoder Pushbutton switch and set Fast/Slow tuning mode
    
    read_volts();  // Read and display the battery voltage
  }
  
  check_key();  // Check the Key and Transmit CW
  
}  // End Main Loop


///////////////////////////////////////////////////////////////////////////////
//
//    Latch dit and/or dah press
//
//    Called by keyer routine
//
///////////////////////////////////////////////////////////////////////////////

void update_PaddleLatch()
{
    if (digitalRead(key_dit) == LOW) {
        keyerControl |= DIT_L;
    }
    if (digitalRead(key_dah) == LOW) {
        keyerControl |= DAH_L;
    }

}

///////////////////////////////////////////////////////////////////////////////
//
//    Calculate new time constants based on wpm value
//
///////////////////////////////////////////////////////////////////////////////

void loadWPM (int wpm)
{
    ditTime = 1200/wpm;
}

void read_rit()
{
  // Read and set the RIT frequency  
  rit = analogRead(rit_pot);  // Read the RIT pot
  if (rit != old_rit)  // Set the RIT if it has changed
  {
    old_rit = rit;
    if (current_step)
    {
      rit_range = rit_fast_step;  // Set the RIT Range for Fast Step
    } else {
      rit_range = rit_slow_step;  // Set the RIT Range for Slow Step
    }
    rit_freq = map(rit,10,1020,rit_range * -1, rit_range);  // Map the RIT frequency to the appropriate range
    if (abs(rit_freq) > rit_range)  // Constrain the RIT to the RIT range if necessary
    {
      if (rit_freq > 0)
      {
        rit_freq = rit_range;
      } else {
        rit_freq = rit_range * -1;
      }
    } 
    if(abs(rit_freq) < rit_range/50)
    {
      rit_freq = 0;
    }
    rx_freq = tx_freq + rit_freq;
    set_DDS(rx_freq);     
    glcd.printNumI(rx_freq,30,8,7); // Update the RX frequency display 
 
  } 
}    

void set_keyer_speed()
{
  if (key_mode !=2)
  {
    // Read and set the CW Keyer Speed
    cw_speed = map(analogRead(speed_Pot),10,1020,5,35);  // Read the CW Speed Pot
    if (cw_speed != old_speed)  // Update if the speed has changed
    {
      old_speed = cw_speed;
//      cw_wpm = map(cw_speed,10,1020,5,35);  // Map the CW Speed pot range to 5-35 wpm
      if (cw_speed > 35)
      {
        cw_speed = 35;
      }
    
      if(cw_speed < 5)
      {
        cw_speed = 5;
      }
      glcd.printNumI(cw_speed,40,32,2); // Update the CW Speed display
      loadWPM(cw_speed);   // Set the keying speed
    }
  }
}  

void read_encoder()
{
  // Read the Encoder  
  encoder_Pos = Enc.read()/4; // divide by 4 to match encoder detent
  if (encoder_Pos != old_encoder_Pos) // If the Encoder has changed update freq
  {
    if (encoder_Pos < old_encoder_Pos) // If we're increasing frequency
    {
      if (tx_freq >= band_high) // Limit to top end of band
      {
        tx_freq = band_high;
      } else {
        tx_freq = tx_freq + step_size;
      }
     
    } else {
      if (tx_freq <= band_low)// We're decreasing frequency, limit to low end of band
      {
        tx_freq = band_low;
      } else {
        tx_freq = tx_freq - step_size;
      }
    }

//    dds.setFrequency(tx_freq); // Set the TX frequency
    old_encoder_Pos = encoder_Pos;
    rx_freq = tx_freq + rit_freq;
    set_DDS(rx_freq);  
    glcd.printNumI(tx_freq,30,0,7);  // display the Tx Freq
    
    // Add in the RIT for the Rx freq
    
    glcd.printNumI(rx_freq,30,8,7);  // display the Tx Freq    
    
  }
}  

void read_pushbutton()
{
  // Read the Encoder Pushbutton Switch
  encoder_Switch = digitalRead(encoder_Switch_Pin);
  if(encoder_Switch == LOW && millis() > delay_time) // Check to see if pressed
  {
    // if it's changed, toggle the step size but don't allow again for debounce time
    delay_time = millis() + debounceInterval; // if it's changed, toggle the step size but don't allow again for debounce time
    current_step = !current_step; // Change the Step rate
    if (current_step)
    {
      glcd.print("Fast",35,24);
      step_size = fast_step;
    } else {
      glcd.print("Slow",35,24);
      step_size = slow_step;
    }
  }
}  

void check_key()
{
  if (key_mode == 2) // Straight Key
  {
    // Straight Key Mode
    if ((digitalRead(key_dit) == LOW) || (digitalRead(key_dah) == LOW))
    {
      keyerState = KEYED;
      set_DDS(tx_freq + tx_offset);
      // Key from either paddle
      digitalWrite(key_pin, HIGH);
    } else {
      set_DDS(rx_freq);      
      digitalWrite(key_pin, LOW);
      keyerState = IDLE;
    }
  } else {
    
  // Basic Iambic Keyer
  // keyerControl contains processing flags and keyer mode bits
  // Supports Iambic A and B
  // State machine based, uses calls to millis() for timing.
  switch (keyerState) 
  {
    case IDLE:      // Wait for direct or latched paddle press
      if ((digitalRead(key_dah) == LOW) || (digitalRead(key_dit) == LOW) || (keyerControl & 0x03)) 
      {
        update_PaddleLatch();
        keyerState = CHK_DIT;
      }
      break;

    case CHK_DIT:      // See if the dit paddle was pressed
      if (keyerControl & DIT_L) 
      {
        keyerControl |= DIT_PROC;
        ktimer = ditTime;
        keyerState = KEYED_PREP;
      } else {
        keyerState = CHK_DAH;
      }
      break;
       
    case CHK_DAH:      // See if dah paddle was pressed
      if (keyerControl & DAH_L) 
      {
        ktimer = ditTime*3;
        keyerState = KEYED_PREP;
      } else {
        keyerState = IDLE;
      }
        break;

    case KEYED_PREP:      // Assert key down, start timing, state shared for dit or dah
      set_DDS(tx_freq + tx_offset);                    // Set the DDS to TX freq
      digitalWrite(key_pin, HIGH);         // Key the TX Relay
      ktimer += millis();                  // set ktimer to interval end time
      keyerControl &= ~(DIT_L + DAH_L);    // clear both paddle latch bits
      keyerState = KEYED;                  // next state
      break;

    case KEYED:      // Wait for timer to expire
      if (millis() > ktimer)  // are we at end of key down ?
      {           
        set_DDS(rx_freq);
        digitalWrite(key_pin, LOW);      // Turn the TX Relay off
        ktimer = millis() + ditTime;     // inter-element time
        keyerState = INTER_ELEMENT;      // next state
      }
      else if (key_mode == 1) // Check to see if we're in Iambic B Mode
      {
        update_PaddleLatch();           // early paddle latch in Iambic B mode
      }
      break;
      break;

    case INTER_ELEMENT:      // Insert time between dits/dahs
      update_PaddleLatch();               // latch paddle state
      if (millis() > ktimer)  // are we at end of inter-space ? 
      {              
        if (keyerControl & DIT_PROC)      // was it a dit or dah ?
        {        
          keyerControl &= ~(DIT_L + DIT_PROC);   // clear two bits
          keyerState = CHK_DAH;                  // dit done, check for dah
        } else {
          keyerControl &= ~(DAH_L);              // clear dah latch
          keyerState = IDLE;                     // go idle
        }
      }
      break;
    } 
  } 
}  

void set_DDS(double set_freq)
{
  dds.setFrequency(set_freq);  // Set the DDS to the starting freq
  if (set_freq >= band_high)
  {
    set_freq = band_high;
  }
  if (set_freq <= band_low)
  {
    set_freq = band_low;
  }
//  glcd.printNumI(set_freq,30,16,7);  // Display the DDS Frequency
}

void read_volts()
{
  int battery = analogRead(battery_pin);
  float battery_voltage = map(battery,0,1023,0,volt_cal);// Map the Battery A/D value to voltage
  glcd.printNumF((battery_voltage/100) + volt_drop,2,35,40); // Display the Battery voltage
  glcd.print("V",65,40);
}  
  
  
