/*************************************************************************
Program to run the AD9951-based frequency synthesizer module being developed by Tom and myself.  Only one half is used in this LQ meter.
The communication to the module is very straightforward and consists of a series of ASCII commands.

In this program, we also need to get external input from a keypad and/or a rotary encoder.


The revised pseudo-code for the program is:

     initialize everything;
     loop
        see if there's something from the serial port; if so service it
	see if there is anything from the rotary encoder; if so service it
	see if the 'select' button has bee pressed; if so, service it
	see if the 'set' scan button has been pressed; if so, service it
	refresh the power level bargraph
    until hell freezes over;
    
     The program has four basic modes. these are selected by the 'select' PB switch press
     
Mode 0: measure L and Q automatically by scanning through from 100 KHz to 100 MHz to find a peak.  The variable C
must be defined as the capacitance being used.  The scan is initiated by pressing the SET button. The rotary encoder does nothing.

Mode 1; the rotary encoder is used to set the value of C.  When the desired value is shown, pressing the SET button
transfers the value into the global variable C

Mode 2: the rotary encoder changes the frequency.  If the SET button is pressed at any time, the
display will show the value of Q at this frequency.

When, in mode 0, the resonant frequency is found, then the inductance, in nH, will be given by:

	L = 2.53303E13 / (C * f * f)
	
	where C is in pF and f is in kHz
	
Mode 3: a calibration mode.  A low-inductance short is placed between the L terminals and then the SET push-button is pressed.  The frequency is then varied between 100 KHz and 100 MHz and the values of the two power meters are read.  They should be the same in absolute R power so this set of measurements gives the offset to be subtracted from the measured power ratio to get the true ratio.
	
	Major hardware modification in February, 2005 removed the two relays and put in a separate RF power meter
permanently on the 'cold' L junction and wired th original RF power meter to the 'hot' LC junction.  AD0 is the 'hot' RF power level while AD1 is the 'cold' RF power level.

April, May, 2005 I replaced the two RF meters with just a single one at the hot junction.  I also
replaced the capacitive voltage divider with a resistive one.

NOTE: With the values I have for Ccoupling, the Q measurement is not going to be very accurate for
frequencies less than 250 KHz.  Also, if Q >= 100 or so, it will also not be too accurate.

22 May, 2005 Started adding code to measure Q at frequency set and capacitor varied to get resonance.

23 May, this worked nicely.  This version was saved in the LQ_meter_mark_III directory.  WHile at it, I amended the lcd.c file slightly to make sure that the number of blocks in the barograph
procedure never got larger than the display size.  I think the thing now is in a finished state.
I just need to very accurately measure the coupling capacitor and the stray capacitance and put them
in as final constants.  They are defined in the #defines up in front of the code.

**************************************************************************/
#include <stdlib.h>
#include <avr/io.h>
#include <math.h>
// NOTE: in lcd.h, the LCD port is defined as being PORT C
#include "lcd.h"
#include "cplx.c"
#include <avr/interrupt.h>
#include <avr/signal.h>
#include <avr/wdt.h>
#include <avr/eeprom.h>
#include <avr/sleep.h>
#include <avr/pgmspace.h>


// CPU clock frequency
#define fcpu 14745600
#define BAUDRATE 19200
// and, interrupt driven serial I/O routines
#include "uart0.c"
#define serial_character_has_come_in rbuflen()

// upper limit for DDS synthesizer
#define UPPER_LIMIT 200000000L
// now, define constants for upper and lower scan frequency locations

#define default_f 10000000L
#define default_C 100;
#define number_of_modes 4
// now define how the power will be displayed
#define bargraph 0
#define dBmilliwatts	1

// the numnber of A/D conversions made for each data point measured
#define NumberAveraged 10

#define T0_ON TCCR0 = 5
#define T0_OFF TCCR0 = 0

#define IDIO PD6
#define ISYNC PD5
#define ICLK PD7

#define clr_IDIO PORTD = (PORTD & ~(1 << IDIO))
#define set_IDIO PORTD = PORTD | (1 << IDIO)
#define clr_ISYNC PORTD = (PORTD & ~(1 << ISYNC))
#define set_ISYNC PORTD = PORTD | (1 << ISYNC)
#define select_button_pressed (PINB & (1 << PB0)) == 0
#define set_button_pressed (PINB & (1 << PB1)) == 0
//
// diagnostic_mode, when defined, causes diagnostic info to be sent out the serial
// port at 19200 Baud
//
// #define diagnostic_mode


/*
C0 is just ln(flow) where flow is the lower limit of the scan
C1 is the stepsize for the logarithmic scan.  It is equal to
ln(fupper/flow)/(N_coarse_steps - 1) where fupper is the upper
frequency limit for the scan.  The values below mean that the
scan goes from 0.1 MHz to 100 MHz
*/
#define twopi 6.2832
#define C0 11.5129255
#define C1 0.0345388
#define C2 0.00345388
#define N_coarse_steps 201
#define N_fine_steps 15

/*
The constants below are the values for the input capacitance and resistance of the AD8307;
Ci and Ri which are 1.2 pF and 1.1K respectively.  The coupling capacitor from the 'hot'
terminal of the RF circuit to the AD8307 is Ccoupling.  I've added an arbitrary 0.1 pF to
the input capacitance of the AD8307 for stray circuit capacitance.
Rinstrument is the instrumental source resistance; in this case, 1.24 Ohms.
*/
  
  
#define Ri 1100.0
// all capacitors specified in pF
#define Ci 1.3
// Ccoupling is the coupling capacitor between the 'hot' junction and the AD8307 power detector
#define Ccoupling 9.5
#define Rinstrument 1.24
#define Cstray 13.0

#define CR 0xd
#define LF 0xa
#define SPACE 0x20

/*************************************************************************

First, the global variables

**************************************************************************/

volatile int16_t temp_C;
volatile uint16_t tick;
volatile uint16_t adc_value;	// this is the value passed back by the A/D conversion interrupt routine
volatile uint32_t frequency;

int16_t error_number, C;
int32_t temporary_number;
uint8_t mode, display_mode, measurement_made;
float used_offset;		// offset value to be used in MeasureValue routine

float B, Value_dB, VoltageRatio, Value_Reference_dBm;
static float stored_offset[N_coarse_steps] __attribute__((section (".eeprom"))) = {
  63.0, 
  63.0,   63.0,   63.0,   63.0,   63.0,   63.0,   63.0,   63.0,   63.0,   63.0, 
  63.0,   63.0,   63.0,   63.0,   63.0,   63.0,   63.0,   63.0,   63.0,   63.0, 
  63.0,   63.0,   63.0,   63.0,   69.4,   75.8,   63.4,   70.2,   71.0,   71.0, 
  71.0,   71.0,   71.0,   71.0,   75.8,   79.0,   79.0,   82.1,   96.0,   96.0, 
  96.0,   96.0,   96.0,   97.0,   99.0,  112.0,  112.0,  112.0,  120.0,  120.0, 
 120.0,  124.0,  126.0,  127.0,  127.0,  127.0,  127.0,  127.0,  127.0,  127.0, 
 127.0,  127.0,  127.0,  130.2,  143.0,  143.0,  152.0,  152.0,  156.0,  156.0, 
 157.8,  158.0,  159.0,  159.0,  159.0,  159.0,  159.0,  153.0,  163.0,  176.0, 
 176.0,  176.0,  176.0,  188.8,  192.0,  192.0,  192.0,  192.0,  192.0,  192.0, 
 192.0,  192.0,  192.0,  192.0,  192.0,  192.0,  192.0,  192.7,  193.0,  195.0, 
 195.0,  197.4,  199.0,  199.0,  199.0,  199.0,  201.4,  207.0,  207.0,  224.0, 
 224.0,  224.0,  224.0,  224.0,  224.0,  224.0,  224.0,  224.0,  224.0,  224.0, 
 224.0,  224.0,  224.0,  224.0,  224.0,  224.0,  225.0,  225.0,  225.0,  227.0, 
 227.0,  227.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0, 
 240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0, 
 240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0, 
 240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0, 
 240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0,  240.0, 
 240.0,  240.0,  240.0,  237.4,  227.0,  227.0,  227.0,  227.0,  227.0,  226.6, 
 225.2,  225.0,  225.0,  225.0,  225.0,  225.0,  224.7,  224.0,  224.0,  224.0
 };

static float Cstore __attribute__((section (".eeprom"))) = 100.0;
 
uint8_t blankline[]  = "                ";

float offset[N_coarse_steps]; 

uint8_t str[32];


/*************************************************************************

Then, the prototypes for the various functions and procedures

**************************************************************************/

void initialize(void);
void send_freq_to_dds(void);			// sends the frequency, frequqncy,  to the DDS
void USART_init(uint16_t baudrate_desired);	// iniitalize the serial port to a particular baud rate
void USART_transmit(uint8_t data);		// sends out the byte, data, to the serial port
uint8_t USART_receive(void);			// gets a byte from the serial port
void wait_three_secs(void);			// routine to wait three seconds - changes nothing except 'tick'
void right_justify(int32_t f, uint8_t separator);// makes 'str' into right justified string for frequency f
int32_t round(double d);			// returns the rounded value of the double
void recover_eeprom(void);			// gets data from the eeprom into the ram storage areas
void get_power(void);				// gets power level from A/D and refreshes the bargraph
uint16_t read_adc(uint8_t channel);		// does A/D conversion of a channel - the result is in adc_value
// the following routines are modified versions of the Codevision routines used in my power meter
float Measure(uint8_t ch);			// returns average value of A/D at channel 'ch'
void MeasureValue(void);			// returns values for globals VoltageRatio ('hot' junction) and
						// Value_dB which is the ratio of hot to cold in dB
void DisplayPower(uint8_t display_mode);	// draws bargraph on bottom line of LCD
void DisplayFrequency(void);			// writes frequency at front of first line
void DisplayC(void);				// writes current value of temp_C at front of first line
void find_frequency(int16_t nl, int16_t nu);		// set global variable, frequency, to the frequency of the peak, do the calculation of L and Q and display them
void doCalibrate(void);				// calibration routine
void USART_write_string(void);			// writes str[] to USART output port
void CalculateQ(void);				// does the actual calculation

/*************************************************************************
**************************************************************************
**************************************************************************
***                                                                    ***
***       Then, the main body of the program                           ***
***                                                                    ***
**************************************************************************
**************************************************************************
**************************************************************************/

int main(void)
  {
  uint8_t ch;
  uint16_t n;
  int16_t nl, nu;
  
  
  initialize();
  lcd_puts("     VA7DIJ\n", 0);
  lcd_puts("    L-Q Meter", 0); 
  wait_three_secs();
   
  recover_eeprom();		// get default data from the eeprom
  B = 0.29231;
     
  temp_C = (uint16_t) C;
  frequency = default_f;
  send_freq_to_dds();
  
  error_number = 0;
  display_mode = bargraph;
  
 /*
 
 After this initalization, we go into the main endless loop where the program checks the four possible sources
 of external interaction:
       serial port,
       rotary encoder
       SELECT button
       SET button
       refresh power bargraph
       
 */ 
 
  for (;;)
    {
   
     wdt_reset();
     

/*
See if there's anything from the serial port
*/     
   if (serial_character_has_come_in)
      {
      ch = UART_getchar();
//      strlcpy_P(str, PSTR("Testing\r\n"), 16);
//      OutStr(str);

      UART_putchar(ch + 1);

      }     
/*
See if there's anything from the rotary encoder - nothing in main program - all done in interrupt routine
*/ 
    
/*
See if the 'select' button has been pressed
*/
    if (select_button_pressed)
      {
      tick = 0;
      while (tick < 2)  wdt_reset();
      measurement_made = 0;
      
      // now, the switch has been debounced
      // so, do the action
      
     mode++;
     mode %= number_of_modes;
     
     if (mode == 1) temp_C = C;
     
      // then, wait till the button is released
      
      while (select_button_pressed) wdt_reset();
      // and debounce
      tick = 0;
      while (tick < 2) wdt_reset();
      }
  
/*
See if the 'Set' button has been pressed
*/
    if (set_button_pressed)
      {
      tick = 0;
      while (tick < 2)  wdt_reset();
      
      // now, the switch has been debounced
      
      switch (mode)
        {
	case 0: lcd_gotoxy(0, 1, 0);
		lcd_puts(blankline, 0);
		lcd_gotoxy(0, 0, 0);
		lcd_puts(blankline, 0);
		lcd_gotoxy(0, 0, 0);
		lcd_puts("Measuring...", 0);
	        find_frequency(0, N_coarse_steps);
		measurement_made = 1;
	        break;
		
	case 1: lcd_gotoxy(0, 1, 0);
		lcd_puts(blankline, 0);
		C = temp_C;
		eeprom_busy_wait();
		eeprom_write_block(&C, &Cstore, 4);
	        measurement_made = 0;
	        break;
		
	case 2: mode = 4;
	        break;
	
	case 3: lcd_gotoxy(0, 0, 0);
		lcd_puts(blankline, 0);
		lcd_gotoxy(0, 0, 0);
		lcd_puts("Calibrating...", 0);
		doCalibrate();
		measurement_made = 0;
		break;
		
	case 4: C = temp_C;
		eeprom_busy_wait();
		eeprom_write_block(&C, &Cstore, 4);
		lcd_gotoxy(0, 1, 0);
		lcd_puts(blankline, 0);
		lcd_gotoxy(0, 0, 0);
		lcd_puts(blankline, 0);
		lcd_gotoxy(0, 0, 0);
		lcd_puts("Measuring...", 0);
		nl = ((log(frequency) - C0) / C1) - 5;
		nu = nl + 10;
		if (nl < 0) nl = 0;
		if (nu > N_coarse_steps) nu = N_coarse_steps;
  #ifdef diagnostic_mode
        
    str[0] = 'F';
    str[1] = 'r';
    str[2] = 'e';
    str[3] = 'q';
    str[4] = ' ';
    str[5] = 0;
    USART_write_string();
    dtostrf((double) (frequency), 10, 1, str);
    USART_write_string();
    USART_transmit(CR);
    USART_transmit(LF);
    
    str[0] = 'N';
    str[1] = 'l';
    str[2] = ' ';
    str[3] = 0;
    USART_write_string();
    dtostrf((double) nl, 10, 1, str);
    USART_write_string();
    USART_transmit(CR);
    USART_transmit(LF);
    
    str[0] = 'N';
    str[1] = 'u';
    str[2] = ' ';
    str[3] = 0;
    USART_write_string();
    dtostrf((double) nu, 10, 1, str);
    USART_write_string();
    USART_transmit(CR);
    USART_transmit(LF);
    
    
  #endif
  		
	        find_frequency(nl, nu);
		measurement_made = 1;
		mode = 0;
		break;
	
	
	default: break;
	}
	
      while (set_button_pressed) wdt_reset();
      // and debounce at the end
      tick = 0;
      while (tick < 2) wdt_reset();
     
      }
  
/*
Measure RF level and display it
*/
      switch (mode)
        {
	case 0: if (measurement_made == 0)
	          {
		  lcd_gotoxy(0, 1, 0);
		  lcd_puts(blankline, 0);
		  lcd_gotoxy(0, 0, 0);
		  lcd_puts(blankline, 0);
		  lcd_gotoxy(0, 0, 0);
		  lcd_puts("Measure L", 0);
		  tick = 0;
		  while (tick < 10) wdt_reset();
		  }
		break;
	
	case 4:
	case 1: lcd_gotoxy(0, 1, 0);
		lcd_puts(blankline, 0);
		DisplayC();
	        tick = 0;
		while (tick < 2) wdt_reset();
                break;
	
	case 2: n = (uint16_t) ((log(frequency) - C0) / C1);
		if (n < 0) n = 0;
		if (n >= N_coarse_steps) n = N_coarse_steps - 1;
	        used_offset = offset[n];
	        MeasureValue();
	        DisplayFrequency();
                DisplayPower(display_mode);
                break;
		
	case 3: lcd_gotoxy(0, 1, 0);
		lcd_puts(blankline, 0);
		lcd_gotoxy(0, 0, 0);
		lcd_puts(blankline, 0);
		lcd_gotoxy(0, 0, 0);
		lcd_puts("Press SET to cal", 0);
		tick = 0;
		while (tick < 10) wdt_reset();
		break;
		
	default: break;
	}
    }
    
  }

/*************************************************************************

Then, the various functions and procedures themselves

**************************************************************************/

void CalculateQ(void)
// calculates the true Q as defined by the equations in my notebook for this project
// it uses the VoltageRatio (the 'effective Q' of the inductor as measured) and the
// frequency and the value of C to do this.  It also uses the constants of the coupling capacitor,
// the input C and R of the AD8307
  {
  
  float L, omega, Qtrue, Fc, Fm, r;

  struct Complex Za, Zc, Zs, Zt, Zparc, Zt_prime, Zm;
  
  Zs.r = Rinstrument;
  Zs.i = 0.0;
  
  L = 0.0;
  omega = twopi * frequency;
  // below, Za is initially an admittance of the AD8307
  Za.r = 1.0 / Ri;
  Za.i = omega * Ci * 1.0e-12;
  // this next line converts it to an impedance;
  Za = C_inv(Za);
  
  Zc.r = 0.0;
  Zc.i = -1.0e12 / (omega * Ccoupling);
  
  Zt.r = 0.0;
  Zt.i = -1.0e12 / (omega *Cstray);
  
  Zparc = C_div(C_mul(Zt, C_add(Zc, Za)), C_add(C_add(Zt, Zc), Za));
  
  Fc = C_mag(C_div(C_mul(Za, Zparc), C_mul(C_add(Za, Zc), C_add(Zparc, Zs))));
  
  Zt_prime.r = 0.0;
  Zt_prime.i = -1.0e12 / (omega * (Cstray + C));
  
  Zm = C_div(C_mul(C_add(Zc, Za), Zt_prime), C_add(C_add(Zt_prime, Zc), Za));
  
  L = -Zm.i / omega;
  
  Fm = C_mag(C_div(C_mul(Za, Zm), C_add(Za, Zc)));
  
  r = (Fm / (VoltageRatio * Fc)) - (Rinstrument + Zm.r);
   
  Qtrue = omega * L / r;
    
  #ifdef diagnostic_mode
        
    str[0] = 'F';
    str[1] = 'r';
    str[2] = 'e';
    str[3] = 'q';
    str[4] = ' ';
    str[5] = 0;
    USART_write_string();
    dtostrf((double) (omega * 1e-6 / twopi), 10, 1, str);
    USART_write_string();
    USART_transmit(CR);
    USART_transmit(LF);
    
    str[0] = 'Z';
    str[1] = 'p';
    str[2] = 'c';
    str[3] = ':';
    str[4] = ' ';
    str[5] = 0;
    USART_write_string();
    dtostrf((double) (C_mag(Zparc)), 10, 6, str);
    USART_write_string();
    USART_transmit(CR);
    USART_transmit(LF);
    
    str[0] = 'R';
    str[1] = 'p';
    str[2] = 'c';
    str[3] = ':';
    str[4] = ' ';
    str[5] = 0;
    USART_write_string();
    dtostrf((double) (Zparc.r), 10, 6, str);
    USART_write_string();
    USART_transmit(CR);
    USART_transmit(LF);
    
    str[0] = 'C';
    str[1] = 'p';
    str[2] = 'c';
    str[3] = ':';
    str[4] = ' ';
    str[5] = 0;
    USART_write_string();
    dtostrf((double) (-1.0e12 / (omega * Zparc.i)), 10, 6, str);
    USART_write_string();
    USART_transmit(CR);
    USART_transmit(LF);
    
    str[0] = 'F';
    str[1] = 'c';
    str[2] = ':';
    str[3] = ' ';
    str[4] = 0;
    USART_write_string();
    dtostrf((double) (Fc), 10, 6, str);
    USART_write_string();
    USART_transmit(CR);
    USART_transmit(LF);
    
    str[0] = 'F';
    str[1] = 'm';
    str[2] = ':';
    str[3] = ' ';
    str[4] = 0;
    USART_write_string();
    dtostrf((double) (Fm), 10, 6, str);
    USART_write_string();
    USART_transmit(CR);
    USART_transmit(LF);
    
    str[0] = 'V';
    str[1] = 'r';
    str[2] = ':';
    str[3] = ' ';
    str[4] = 0;
    USART_write_string();
    dtostrf((double) VoltageRatio, 6, 1, str);
    USART_write_string();
    USART_transmit(CR);
    USART_transmit(LF);
       
    str[0] = 'R';
    str[1] = 'm';
    str[2] = ':';
    str[3] = ' ';
    str[4] = 0;
    USART_write_string();
    dtostrf((double) Zm.r, 10, 6, str);
    USART_write_string();
    USART_transmit(CR);
    USART_transmit(LF);
  
    str[0] = 'C';
    str[1] = 'm';
    str[2] = ':';
    str[3] = ' ';
    str[4] = 0;
    USART_write_string();
    dtostrf((double) (-1.0e12 /(omega * Zm.i)), 6, 1, str);
    USART_write_string();
    USART_transmit(CR);
    USART_transmit(LF);
    USART_transmit(LF);
  #endif

//  Qtrue = VoltageRatio;
    
  lcd_gotoxy(0, 0, 0);
  lcd_puts(blankline, 0);
  lcd_gotoxy(0, 1, 0);
  lcd_puts(blankline, 0);


  lcd_gotoxy(0, 0, 0);
  lcd_puts("L: ", 0);
  dtostrf((double) (L * 1.0e6), 5, 2, str);
  lcd_puts(str, 0);
  lcd_puts("  F(MHz)", 0);

/*
  lcd_gotoxy(0, 0, 0);
  lcd_puts("Rm: ", 0);
  dtostrf((double) Rm, 5, 2, str);
  lcd_puts(str, 0);
  lcd_gotoxy(0, 1, 0);
  lcd_puts("Cm: ", 0);
  dtostrf((double) Cm * 1.0e12, 5, 2, str);
  lcd_puts(str, 0);
*/  


  if ((Qtrue > 0.0) && (Qtrue < 300.0))
    {
    dtostrf((double) Qtrue, 5, 0, str);
    lcd_gotoxy(0, 1, 0);
    lcd_puts("Q: ", 0);
    lcd_puts(str, 0);
    dtostrf((double) (frequency / 1.0e6), 7, 2, str);
    lcd_puts(str, 0);
    }
  else
    {
    dtostrf((double) VoltageRatio, 5, 0, str);
    lcd_gotoxy(0, 1, 0);
    lcd_puts("Q >>", 0);
    lcd_puts(str, 0);
    dtostrf((double) (frequency / 1.0e6), 7, 2, str);
    lcd_puts(str, 0);
    }
}

void USART_write_string(void)
  {
  uint8_t i;
  
  i = 0;
  while (str[i])
    {
    USART_transmit(str[i]);
    i++;
    }
  }

void doCalibrate(void)
/*
  In this routine, the frequency is swept from 100 KHz to 100 MHz and the RF level is measured at each
frequency.  This is done when a low-inductance strap is put in place of the inductor.

Then, the measured value of the ratio at any frequency is measured ratio(dBm) - offset.
*/
  {
  uint8_t i;
  float v0;
  
  for (i = 0; i < N_coarse_steps; i++)
    {
    frequency = exp(C0 + (C1 * i));
    send_freq_to_dds();
    tick = 0;
    while (tick < 1) wdt_reset();
    
    v0 = Measure(0);
    offset[i] = v0;
    eeprom_busy_wait();
    eeprom_write_block(&offset[i], &stored_offset[i], 4);
    
    #ifdef diagnostic_mode
      
      dtostrf((double) v0, 6, 1, str);
      USART_write_string();
      USART_transmit(','); USART_transmit(SPACE);
    
      if ((i % 10) == 0)
        {
        USART_transmit(CR);
        USART_transmit(LF);
        }

    #endif
    
    wdt_reset();    
    }
  }
  
void DisplayC(void)
  {
  right_justify(temp_C, ' ');
  lcd_gotoxy(0, 0, 0);
  lcd_puts(blankline, 0);
  lcd_gotoxy(0, 0, 0);
  lcd_puts(str, 0);
  lcd_puts(" pF", 0);  
  }

void DisplayFrequency(void)
// displays the frequency, in KHz at the front of the first line of the LCD display
  {
  right_justify(frequency / 1000.0, '.');		// right justified string is now in str
  lcd_gotoxy(0, 0, 0);
  lcd_puts(blankline, 0);
  lcd_gotoxy(0, 0, 0);
  lcd_puts(str, 0);
  lcd_puts(" MHz", 0);
  }			// calculates the constants of best fit

void find_frequency(int16_t nl, int16_t nu)
/*
	The program scans between 100 KHz and 100 MHz in 200 equally spaced (logarithmically) steps
and detects the maximum in that range.  Then, around that point, in 15 ten-times smaller steps
to narrow in on the actual peak.

	The expression for the frequency is:
	
	f = exp (11.5129255 + (Nc * 0.0345388) + (Nf * 0.00345388)) where Nc is the coarse
step number and Nf is the fine step number.
*/
  {
  
  
  
  float temp, max;
  int16_t i, Ncp, Nfp, Nlow, Nhigh;
  
  
  max = 0.0;
  Ncp = 0;
  Nfp = 0;
 
  for (i = nl; i < nu; i++)
    {
    frequency = exp (C0 + (C1 * i));
    send_freq_to_dds();			// sets the dds to the global variable, frequency
    used_offset = offset[i];
    MeasureValue();
    if (VoltageRatio > max)
      {
      Ncp = i;
      max = VoltageRatio;
      }
    }
      
// at this point, Ncp contains the maximum frequency found during the coarse steps

  temp = C0 + (C1 * Ncp);
  used_offset = offset[Ncp];
  max = 0.0;
  
  Nlow = - N_fine_steps;
  Nhigh =  N_fine_steps;
  
  if ((Nhigh + Ncp) > N_coarse_steps) Nhigh = N_coarse_steps - Ncp;
  if (Ncp + Nlow < 0) Nlow = -Ncp;
  
  for (i = Nlow; i < Nhigh; i++)
    {
    frequency = exp (temp + (C2 * i));
    send_freq_to_dds();
    MeasureValue();
    if (VoltageRatio > max)
      {
      Nfp = i;
      max = VoltageRatio;
      }
    }
    
  frequency = exp (C0 + (C1 * Ncp) + (C2 * Nfp));
  send_freq_to_dds();
  MeasureValue();
  
// now, calculate and display the measured values

  CalculateQ();
  }


void DisplayPower(uint8_t display_mode)
  {

  if (display_mode == bargraph) lcd_barograph(VoltageRatio, 1, 0);
  else
    {
    lcd_gotoxy(0, 1, 0);
    lcd_puts(blankline, 0);
    lcd_gotoxy(0, 1, 0);
    if (VoltageRatio < 0.0)
      {
      lcd_puts("-", 0);
      VoltageRatio = 0.0 - VoltageRatio;
      }
    right_justify(round(VoltageRatio * 1000.0), '.');
    lcd_puts(str, 0);
    lcd_puts(" ratio", 0);
    }
  
  }
  
int32_t round(double d)
  {
  int32_t temp;
  
  temp = d;
  if ((d - temp) >= 0.5) d++;
  return d;
  }
    
float Measure(uint8_t ch) {

// takes NumberAveraged A/D readings and returns the average of them

  float sum;
  int n;

  tick = 0;
  while (tick == 0) wdt_reset();	// wait one mS for output of detector to settle
  
  sum = 0.0;
  for (n = 0; n < NumberAveraged; n++) sum += (float) read_adc(ch);
  sum /= (float) NumberAveraged;
 
  return sum;
  }
  
void MeasureValue(void) {
//
// sets the global variables, Value_dB and VoltageRatio
// where VoltageRatio is the voltage at the 'hot' junction and Value_dB is the ratio of 'hot' to 'cold' in dB
//  
  float v0;
  
  v0 = Measure(0);
  Value_dB = B * (v0 - used_offset);		// this is the difference in dB
  VoltageRatio = pow(10.0, Value_dB / 20.0);	// root of power ratio so this is the voltage ratio
  }

 uint16_t read_adc(uint8_t channel)
   {
   uint8_t i;
   /*
   Below is the single conversion process without using interrupts
   */
/*   ADCSRA = (1 << ADEN);
   ADMUX = channel & 0x7;
   ADCSRA |= (1 << ADSC);
   while (ADCSRA & (1 << ADSC)) wdt_reset();
   return ADCW;
*/
 // turn off timer so don't get timer interrupts
  T0_OFF;
  
  ADMUX = channel & 0x07;			// select channel 0, external Vref
  
  set_sleep_mode(SLEEP_MODE_ADC);		// enable the ADC noise reduction sleep mode
  
  i = 0;
  while (i++ < 100) wdt_reset();		// give the A/D mux a few microsec to settle
  
  sleep_mode();
// it gets here when the ADC is complete and adc_value now contains the value of  A/D conversion
  T0_ON;					// turn T0 timer back on
  return adc_value;

  }
    
void recover_eeprom(void)
// gets data from EEPROM and puts it into the RAM storage area
  {
  
  eeprom_busy_wait();
  eeprom_read_block(&offset[0], &stored_offset[0], 804);
  eeprom_busy_wait();
  eeprom_read_block(&C, &Cstore, 4);
   
  }

  

void right_justify(int32_t f, uint8_t separator)
// makes a right justified string out of the frequency, f
// the string is put into 'str'

#define length_of_frequency_display 7
  {
  
  uint8_t s[length_of_frequency_display];
  uint8_t i, j, k;
  
  if (f == 0)
    {
    s[0] = '0';
    i = 1;
    }
  else
    {
    i = 0;
    while (f)
      {
      s[i] = f % 10;
      if (s[i] <= 9) s[i] += 48; else s[i] += 55;	// makes ASCII character
      f /= 10;
      i++;
      }
    s[i] = 0;						// terminates the string
    }
    
  for (j = 0; j < length_of_frequency_display; j++) str[j] = ' ';
  str[j] = 0;						// makes 'str' a string of length_of_frequency_display spaces
  
  k = 0;
  for (j = 0; (j - k) < i; j++)
    {
    str[(length_of_frequency_display - 1) -j] = s[j - k];
    
    if (((j - k) == 2) || ((j - k) == 5))
      {
      k++;
      j++;
      str[(length_of_frequency_display - 1) - j] = separator;
      }
    }
  }

void wait_three_secs(void)
// waits three seconds
  {
  tick = 0;
  while (tick < 300) wdt_reset();
  }
  

void USART_init(uint16_t baudrate)
  {
  uint16_t baudword;	// CPU clock speed in Hz
  
  baudword = (uint16_t) ((fcpu/baudrate)/16) - 1;
  
  UBRRH = (uint8_t) baudword >> 8;
  UBRRL = (uint8_t) baudword & 0xff;
  
  UCSRB = (1<<RXEN) | (1<<TXEN);
  }
  
void USART_transmit(uint8_t data)
  {
  while ((UCSRA & (1<<UDRE)) == 0);
  
  UDR = data;
  }  
uint8_t USART_receive(void)
  {
  if ((UCSRA & (1<<RXC)) == 0) return 0; else return UDR;
  }


    
void clock_a_bit(uint8_t ch)
  {

  tick = 0;
  
  while (PIND & (1 << ICLK)) if (tick >= 2) break;		// wait till ICLK is low; i.e., not busy
  
  if (ch == 0) clr_IDIO; else set_IDIO;
 
  clr_ISYNC;	// clear ISYNC
  
  while ((PIND & (1 << ICLK)) == 0) if (tick >= 4) break;	// wait till ICLK goes high indicating acknowledgement
      
  if (tick >= 2) error_number++;
  
  set_ISYNC;		// set ISYNC 
  clr_IDIO;				// set IDIO low;
  }

void send_byte(uint8_t ch)

  {
  int8_t i;
  
  for (i = 7; i >= 0; i--) clock_a_bit(ch & (1 << i));
  
  }
  
void send_freq_to_dds(void)
  {
  uint8_t tt;
  uint16_t sum;
  	  
  sum = 0x04;
  	    
  send_byte((uint8_t) sum & 0xff);		// send the first byte of the command to the DDS
  	  
  tt = (uint8_t) (frequency >> 24) & 0xff;	// MSByte of frequency to be sent
  sum += tt;
  send_byte(tt);
  tt = (uint8_t) (frequency >> 16) & 0xff;	// 2nd MSByte of frequency to be sent
  sum += tt;
  send_byte(tt);
  tt= (uint8_t) (frequency >> 8) & 0xff;	// 3rd MSByte of frequency to be sent
  sum += tt;
  send_byte(tt);
  tt = (uint8_t) frequency & 0xff;		// LSByte of frequency to be sent
  sum += tt;
  send_byte(tt);
// finally, send the checksum
  send_byte((uint8_t) sum & 0xff);

  
  if (error_number > 0)
    {
    lcd_clrscr(0);
    lcd_puts("Errors: ", 0);
    utoa(error_number, str, 10);
    error_number = 0;
    lcd_puts(str, 0);
    
    tick = 0;
    while (tick < 100);		// wait a sec.
    lcd_clrscr(0);
    lcd_gotoxy(0, 1, 0);
    }
  
  }
  

void initialize(void)
//
// just sets up the ports and the interrupts, etc
//
// PORT C is already setup in lcd.h to be used for the LCD display
// PORT B will be used for the two PB switches
// PORT D top three bits will be used to communicate with the AD9951 controller
//    PD4 is the relay output bit; when 0, selects the LC junction; when high, selects the reference
//
//
  {
  
  DDRA = 0;
  DDRB = 0;
  DDRD = (1 << IDIO) | (1 << ISYNC) | (1 << PD4);		// set IDIO, ISYNC and PD4 lines as an output
  PORTD = PORTD | (1 << ICLK);					// set ICLK line pull-up resistor
  PORTB = (1 << PB0) | (1 << PB1);				// sets pull-ups on PB0 and PB1

  
//
// now, for the timer interrupts.  We'll use T0 as the general purpose 'tick' timer
  
  OCR0 = 144;			// output compare register to give 10 mS 'ticks'
  
  T0_ON;			// set T0 clock to fcpu divided by 1024 which is 14.4 KHz
  
  PORTB = 0xe0;
  
  TIMSK =  1 << OCIE0;
  
  clr_IDIO;
  set_ISYNC;
  

  MCUCR = (1 << ISC01);			// INT0 occurs on falling edge
  GICR = (1 << INT0);			// ENABLE INT0
  
  wdt_enable(WDTO_2S);
  
//  USART_init(BAUDRATE);
  UART_first_init();
  
  ADCSRA = (1 << ADEN) | (1 << ADIE) | (1 << ADPS2) | (1 << ADPS1);		// enable ADC interrupts and A/D converter
  
  lcd_init(LCD_DISP_ON, 0);
    
  sei();
}
 
 /*************************************************************************

Then, the interrupt routines

**************************************************************************/


SIGNAL(SIG_INTERRUPT0)
//
// Interrupt is caused by falling edge of the B ouput of the rotary encoder.  A output, which goes to 
// pin PD3 is either a 0 or a 1 depending on which way the encoder is rotated.
//

  {
  
  switch (mode)
  
    {
    case 0:
            break;
    
    case 4:
    case 1: if (PIND & (1 << PD3)) temp_C--; else temp_C++;
            if (temp_C < 0) temp_C = 0;
            break;
	    
    case 2:
    case 3: if (PIND & (1 << PD3)) frequency = (uint32_t) (0.9950248 * frequency); else frequency = (uint32_t) (1.005* frequency);
            if (frequency > UPPER_LIMIT) frequency = UPPER_LIMIT;
	    send_freq_to_dds();
            break;
    
    default: break;
  
    }
  }



SIGNAL(SIG_OUTPUT_COMPARE0)
  {
  TCNT0 = 0;
  tick++;
  }

SIGNAL (SIG_ADC)
  {
  adc_value = ADCW;
  }
  