#include "uart0.h"
 
/* Buffersizes must be 2^n */

#define TBUFSIZE	64
#define RBUFSIZE	64

#define TMASK		(TBUFSIZE-1)					//Transmit bit-mask
#define RMASK		(RBUFSIZE-1)					//Receive bit-mask

volatile unsigned char tbuf[TBUFSIZE];			//transmit buffer
volatile unsigned char rbuf[RBUFSIZE];			//receive buffer

volatile unsigned char t_in = 0;
volatile unsigned char t_out = 0;

volatile unsigned char r_in = 0;
volatile unsigned char r_out = 0;

/* When modifing this code remember there is a difference between SIGNAL and
   INTERRUPT, the first instruction in the SIGNAL routine is 'cli' thus disabeling
   all interrupts while executing the ISR. INTERRUPT's first instruction is 'sei'
   allowing further interrupt processing */

SIGNAL(SIGNAL_RXCOMPLETE) 
{
	/* Receive interrupt */
	unsigned char c;
	
	c = UDR0;							// Get received char

	rbuf[r_in] = c;
	r_in++;
	r_in &= RMASK;
}

SIGNAL(SIGNAL_TXCOMPLETE) 
{
	/* Data register empty indicating next char can be transmitted */
	if(t_in != t_out) 						//if we didnt transmit all data yet
	{										//send a byte to the data reg.
		UDR0 = tbuf[t_out &TMASK];
		t_out++;	
	}
	else 									//if we did send it all, disable the empty interrupt.
	{
		RXTXEN0_REG &= ~(1 << UDRIE);
	}
}

char tbuflen(void) 
{
	return(t_in - t_out);
}

int UART_putchar(char c) 
{
	/* Fills the transmit buffer, if it is full wait */ 
	while((TBUFSIZE - tbuflen()) <= 2) wdt_reset();
	
	/* Add data to the transmit buffer, enable TXCIE */
	tbuf[t_in & RMASK] = c;
	t_in++;
	
	RXTXEN0_REG |= (1<<UDRIE);			// Enable UDR empty interrupt
	
	return(0);
}

char rbuflen(void) 
{
	return(r_in - r_out);
}

int UART_getchar(void) 
{
	unsigned char chCharacter;

	while(rbuflen() == 0) wdt_reset();
	
	chCharacter = rbuf[r_out];
	r_out++;
	r_out &= RMASK;
	
	return((int) chCharacter );
}

void UART_setbaudrate(unsigned long int ulBaudRate)
{

	unsigned int		BaudRateValue;
	
	BaudRateValue = (unsigned int) (fcpu/ (16 * ulBaudRate)) - 1;
	
	BAUD0H_REG = (unsigned char)(BaudRateValue >> 8);
   	BAUD0L_REG = (unsigned char)(BaudRateValue & 0x00ff);

//  UBRRL = 23;
}

void OutStr(uint8_t *s)
  {
  while (*s)
    {
    UART_putchar(*s);
    s++;
    }
  }

void UART_first_init(void) 
{
	/* First init for the UART */
	// #define BAUD_RATE0_REG  (unsigned int)(CPU_CLK_SPEED / (16 * BAUD_RATE0) ) - 1
	
	UART_setbaudrate(BAUDRATE);

	RXTXEN0_REG = (1 << RXCIE) | (1 << UDRIE) | (1<<TX0EN)|(1<<RX0EN);			// 8 Databits, receive and transmit enabled, receive and transmit complete interrupt enabled
	UDR0 = 0;
}




