

//    Copyright  2010, 2011 Thomas C. McDermott, N5EG
//    This file is part of ABCDmatrix - the 2-Port Network Calculator program.
//
//    ABCDmatrix is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    ABCDmatrix is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with ABCDmatrix, if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//


using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Printing;
using System.IO;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Text;
using System.Windows.Forms;
using System.Windows.Forms.DataVisualization.Charting;


namespace ABCDmatrix
{
    public partial class Form1 : Form
    {

        public List<Tile> TileContainer;    // Holds the network element blocks
        public AnalysisParameter AnParm;           // Holds analysis parameters
        public List<NodeState> NodeList;     // Holds the V,I node voltages and currents at one frequency
        public Tile Clipboard;              // Holds tile that has been cut or copied - initially null

        public SplashForm s;                // Splash screen form
        Timer SplashT;                      // Splash screen display timer

        String FileIdentString = "ABCDmatrix Save File Version 1.0.0"; // Identify the saved file type & revision

        /// <summary>
        /// Main display form
        /// </summary>
        public Form1()
        {
            try
            {
                InitializeComponent();
            }
            catch (FileNotFoundException ex)    
            {
                    MessageBox.Show("Error: Cannot initialize the ABCDMatrix application.\n\n" +
                        "Have you installed the   Microsoft Chart Control ?\n\n" +
                        "Missing File/Assembly: " + ex.FileName, 
                        "File Not Found - IO error",
                        MessageBoxButtons.OK, MessageBoxIcon.Stop);

                    throw(ex);
            }
            
           
            s = new SplashForm();       // Display the splash screen
            s.Show();

            // Start a timer to close the splash screen after delay
            SplashT = new Timer();
            SplashT.Interval = 6000; // 6 seconds to show the splash screen
            SplashT.Tick += new EventHandler(SplashTimerExpire);
            SplashT.Start();
           

            // TileContainer holds the network design as a collection of network elements (tiles)
            TileContainer = new List<Tile>();

            // AnParm holds the analysis attributes 
            AnParm = new AnalysisParameter();

            // Hold all the node voltages and currents (one more than the number of tiles)
            NodeList = new List<NodeState>();

            // set frequency range from AnParm
            XMintextBox.Text = AnParm.StartFrequency.ToString();
            XMaxtextBox.Text = AnParm.StopFrequency.ToString();

            // set start and stop analysis node numbers
            FromNodetextBox.Text = AnParm.FromNode.ToString();
            ToNodetextBox.Text = AnParm.ToNode.ToString();
        }

        // Close and dispose of the Splash Screen
        // Shutdown and dispose of the Splash Timer
        private void SplashTimerExpire(object sender, EventArgs e)
        {
            SplashT.Stop();
            s.Close();

            SplashT.Tick -= SplashTimerExpire;
            SplashT.Dispose();
            s.Dispose(); 
        }

        /// <summary>
        /// Paint the network design page (Tab1)
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void PaintDesignTab(object sender, PaintEventArgs e)
        {
            Tile.BackgroundNetDraw(e.Graphics);  // draw the background grid
            Tile.NetElementDraw(e.Graphics, TileContainer); // draw the tiles within TileContainer
        }

        /// <summary>
        /// Print the network design page (Tab)
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="ev"></param>
        private void PrintDesignTab(object sender, PrintPageEventArgs ev)
        {
            Tile.BackgroundNetDraw(ev.Graphics);  // draw the background grid
            Tile.NetElementDraw(ev.Graphics, TileContainer); // draw the tiles within TileContainer
        }

        /// <summary>
        /// Check for MouseUp event on the network design page (Tab1).
        /// Left = Tile selection, Right = Add/Delete action menu launch
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void DesignTabMouseUp(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
                if (Tile.HitSelect(e, TileContainer))       // Select the tile if it was hit
                {
                    Tile tile = Tile.SelectedTile(TileContainer);
                    propertyGrid1.SelectedObject = tile;    // set the property grid browser to the selected tile
                    Refresh();                             // and refresh if selection changed
                }

            if (e.Button == MouseButtons.Right)
            {
                // context menu asking for insert to left, insert to right, or delete
                contextMenuStrip1.Show(tabPage1, e.X, e.Y);
            }

        }

        /// <summary>
        /// User requested Insert Left from right-click menu.
        /// Get index of selected tile, insert new tile before user selected tile.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void InsertNetLeft(object sender, EventArgs e)
        {
            int index = Tile.SelectedIndex(TileContainer);
            if (index == -1)
                index = 0;          // no tile selected - insert tile at front
            InsertNewTile(index);
        }

        /// <summary>
        /// User requested Insert Right from right-click menu.
        /// Get selected tile, insert new tile after user selected tile.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void InsertTileRight(object sender, EventArgs e)
        {
            int index = Tile.SelectedIndex(TileContainer);
            InsertNewTile(index + 1);
        }

        /// <summary>
        /// User request Delete from right-click menu.
        /// Delete tile that has been selected by user mouse action.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void DeleteTile(object sender, EventArgs e)
        {
            int index = Tile.SelectedIndex(TileContainer);
            if (index >= 0)                                 // if a tile is selected
            {
                TileContainer.RemoveAt(index);              // delete it
                Refresh();
            }
        }

        /// <summary>
        /// Insert new tile at index, ask user to describe it.
        /// </summary>
        /// <param name="index"></param>
        private void InsertNewTile(int index)
        {
            Tile newtile = new Tile(NetworkType.Empty);

            TileInputForm tif = new TileInputForm(newtile);
            if (tif.ShowDialog() == DialogResult.OK)    // User selected a tile & type
            {

                if (newtile.BlockType == NetworkType.ABCDparam ||
                    newtile.BlockType == NetworkType.Sparam ||
                    newtile.BlockType == NetworkType.Zparam ||
                    newtile.BlockType == NetworkType.Yparam)
                {
                    if (newtile.ParamType == ParameterType.Fixed)
                    {
                        // tile is kind with 4 fixed matrix elements - fill in values
                        FixedParameterentry fe = new FixedParameterentry(newtile);
                        if (fe.ShowDialog() == DialogResult.Cancel)
                            return;         // return without accepting the tile
                    }
                    else                    // tile is kind with a frequency-dependent list of values.
                    {
                        // Get the filespec.
                        OpenFileDialog spec = new OpenFileDialog();
                        spec.Filter = "parameter list (*.s2p)|*.s2p";
                        spec.CheckFileExists = true;

                        // Open the file
                        if (spec.ShowDialog() == DialogResult.OK)
                        {
                            // Read variable number of parameters from the S2P file into the tile
                            newtile.VariableParameterFileName = spec.FileName;
                            if (Tile.ReadS2PFile(newtile, spec.FileName) == false)
                                return;     // read of S2P failed, don't accept the tile
                        }
                        else                // did not sucessfully get file, don't accept the tile
                        {
                            newtile.VariableParameterFileName = null;
                            return;
                        }
                    }
                }

                // Valid tile entered and user said OK

                foreach (Tile tile in TileContainer)  // de-select all tiles
                    tile.Select = false;

                newtile.Select = true;                   // select the tile user just entered
                TileContainer.Insert(index, newtile);      // insert tile
                propertyGrid1.SelectedObject = newtile;    // set the property grid browser to the selected tile

                Refresh();
            }
        }

        // cut selected tile, save copy of tile reference on the clipboard
        private void CutTile(object sender, EventArgs e)
        {
            int index = Tile.SelectedIndex(TileContainer);
            if (index >= 0)
            {
                Clipboard = Tile.SelectedTile(TileContainer);
                TileContainer.RemoveAt(index);
                Refresh();
            }
            else
                MessageBox.Show("Nothing to Cut", "Can't Cut",
                    MessageBoxButtons.OK, MessageBoxIcon.Information);
        }

        // copy selected tile reference to the clipboard
        private void CopyTile(object sender, EventArgs e)
        {
            int index = Tile.SelectedIndex(TileContainer);
            if (index >= 0)
                Clipboard = Tile.SelectedTile(TileContainer);
            else
                MessageBox.Show("Nothing to Copy", "Can't Copy",
                    MessageBoxButtons.OK, MessageBoxIcon.Information);
        }

        private void PasteTileLeft(object sender, EventArgs e)
        {
            if (Clipboard != null)
            {
                int index = Tile.SelectedIndex(TileContainer);  //if there's a tile on the clipboard
                if (index == -1)
                    index = 0;          // no tile selected, insert tile at front
                TileContainer.Insert(index, new Tile(Clipboard));      // insert new tile
                Refresh();
            }
            else
                MessageBox.Show("No tile on the Clipboard", "Can't Paste",
                    MessageBoxButtons.OK, MessageBoxIcon.Information);
        }

        private void PasteTileRight(object sender, EventArgs e)
        {
            if (Clipboard != null)
            {
                int index = Tile.SelectedIndex(TileContainer);  //if there's a tile on the clipboard
                if (index == -1)
                    index = 0;          // no tile selected, insert tile at front
                TileContainer.Insert(index+1, new Tile(Clipboard));      // insert new tile
                Refresh();
            }
            else
                MessageBox.Show("No tile on the Clipboard", "Can't Paste",
                    MessageBoxButtons.OK, MessageBoxIcon.Information);
        }

        /// <summary>
        /// Get filespec and open the file. Read in the tiles.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void FileOpen(object sender, EventArgs e)
        {
            IFormatter formatter = new BinaryFormatter();
            OpenFileDialog openfile = new OpenFileDialog();
            openfile.Filter = "tile (*.til)|*.til";

            if (openfile.ShowDialog() == DialogResult.OK)
            {
                TileContainer.Clear();  // Empty the TileContainer before reading in tiles

                try
                {
                    using (Stream instream = openfile.OpenFile())
                    {
                        if (instream != null)
                        {
                            String FileDescString = formatter.Deserialize(instream) as String;

                            if (FileDescString == null)      // no identifier string found heading the file
                                FileDescString = "No identifier found";

                            if (String.Compare(FileDescString, FileIdentString) != 0)
                            {
                                String errormessage = "File is wrong type or revision.\n\nExpecting: " +
                                    FileIdentString + "\nFile Ident is: " + FileDescString;

                                MessageBox.Show(errormessage, "Can't read Network File",
                                    MessageBoxButtons.OK, MessageBoxIcon.Error);

                                return;
                            }

                            int tilecount = (int)formatter.Deserialize(instream);  // number of tiles in the file
                            // read in all tiles
                            for (int i = 0; i < tilecount; i++)
                            {
                                Tile tile = (Tile)formatter.Deserialize(instream);
                                tile.Select = false;  // deselect each tile
                                TileContainer.Add(tile);
                            }

                            // read in the analysis parameters
                            AnParm = (AnalysisParameter)formatter.Deserialize(instream);

                            instream.Close();
                        }
                    }
                }
                catch (SerializationException ex)
                {
                    MessageBox.Show("Warning: Could not read file from disk. " + ex.Message, "File read error",
                        MessageBoxButtons.OK, MessageBoxIcon.Warning);
                }


                // set frequncy range from AnParm
                XMintextBox.Text = AnParm.StartFrequency.ToString();
                XMaxtextBox.Text = AnParm.StopFrequency.ToString();

                Refresh();      // update the display surface with new design
            }
        }

        /// <summary>
        /// Get filespec, save all the tiles
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void FileSave(object sender, EventArgs e)
        {
            IFormatter formatter = new BinaryFormatter();
            SaveFileDialog savefile = new SaveFileDialog();
            savefile.Filter = "tile (*.til)|*.til";

            if (savefile.ShowDialog() == DialogResult.OK)
            {
                try
                {
                    using (Stream outstream = savefile.OpenFile())
                    {
                        if (outstream != null)
                        {
                            formatter.Serialize(outstream, FileIdentString);    // Type & Revision of file

                            formatter.Serialize(outstream, TileContainer.Count);    // number of tiles saved

                            foreach (Tile tile in TileContainer)
                                formatter.Serialize(outstream, tile);      // save the tiles to the file

                            formatter.Serialize(outstream, AnParm);         // save the analysis parameters
                            outstream.Close();
                        }
                    }
                }
                catch (SerializationException ex)
                {
                    MessageBox.Show("Error: Could not write file to disk. Original error: " + ex.Message);
                }
            }
        }

        /// <summary>
        /// Start a new sheet - erase all the tiles in the design.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void FileNew(object sender, EventArgs e)
        {
            // Warn user first
            if (MessageBox.Show("This will delete all tiles in the current design.\n\rIs it OK?", "New Design",
                MessageBoxButtons.YesNo, MessageBoxIcon.Warning) == DialogResult.Yes)
            {
                TileContainer.Clear();  // delete all tiles from the design
                Clipboard = null;       // empty the clipboard
                Refresh();
            }
        }

        /// <summary>
        /// Display the Analysis browser tab.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void PaintAnalysisTab(object sender, PaintEventArgs e)
        {
            propertyGrid2.SelectedObject = AnParm;
        }

        /// <summary>
        /// Analyzer the network and display the results.
        /// </summary>
        private void Analyze()
        {

            // Analyse the network and display the results.
            // 1. Setup up the chart.
            // 2. Run the analysis and populate the datasets of the chart.

            Double Frequency;       // measurement freuqency
            Int32 index;            // index corresponding to frequency
            ParameterSet P;         // parameters at a frequency
            Boolean TraceErrorFlag = false; // true means some trace(s) can't be displayed


            // If there are no tiles, then don't do any analysis
            if (TileContainer.Count == 0)
                return;

            NodeList.Clear();       // Remove all nodes from the list of V,I measurements

            Chart1.SuppressExceptions = true;   // stop chart from throwing non-critical exceptions

            // Set series chart type
            Chart1.Series["Series1"].ChartType = SeriesChartType.Line;
            Chart1.Series["Series2"].ChartType = SeriesChartType.Line;
            Chart1.Series["Series3"].ChartType = SeriesChartType.Line;
            Chart1.Series["Series4"].ChartType = SeriesChartType.Line;
            Chart1.Series["Series5"].ChartType = SeriesChartType.Line;
            Chart1.Series["Series6"].ChartType = SeriesChartType.Line;
            Chart1.Series["Series7"].ChartType = SeriesChartType.Line;
            Chart1.Series["Series8"].ChartType = SeriesChartType.Line;
            Chart1.Series["Series9"].ChartType = SeriesChartType.Line;

            // Enable X axis margin if true
            Chart1.ChartAreas["Default"].AxisX.IsMarginVisible = false;

            // Show as 3D
            //chart1.ChartAreas["Default"].Area3DStyle.Enable3D = true; 

            Chart1.ChartAreas["Default"].AxisX.Minimum = AnParm.StartFrequency;
            Chart1.ChartAreas["Default"].AxisX.Maximum = AnParm.StopFrequency;

            //chart1.ChartAreas["Default"].AxisY2.Minimum = 180;  // degrees
            //chart1.ChartAreas["Default"].AxisY2.Maximum = -180;

            //chart1.ChartAreas["Default"].AxisY.Minimum = -1E9;  // impedance
            //chart1.ChartAreas["Default"].AxisY.Maximum = 1E9;

            Chart1.Series["Series1"].Points.Clear();
            Chart1.Series["Series2"].Points.Clear();
            Chart1.Series["Series3"].Points.Clear();
            Chart1.Series["Series4"].Points.Clear();
            Chart1.Series["Series5"].Points.Clear();
            Chart1.Series["Series6"].Points.Clear();
            Chart1.Series["Series7"].Points.Clear();
            Chart1.Series["Series8"].Points.Clear();
            Chart1.Series["Series9"].Points.Clear();

            if (AnParm.InputZFormat == ComplexFormat.MagnitudePhase)
            {
                Chart1.Series["Series1"].LegendText = "Input Impedance Magnitude, ohms";
                Chart1.Series["Series2"].LegendText = "Input Impedance Phase Angle, degrees";
            }
            else
            {
                Chart1.Series["Series1"].LegendText = "Input Resistance, ohms";
                Chart1.Series["Series2"].LegendText = "Input Reactance, ohms";
            }

            Chart1.Series["Series3"].LegendText = "Network Voltage Transfer magnitude in dB.";
            Chart1.Series["Series4"].LegendText = "Network Voltage Transfer Phase Angle, degrees";
            Chart1.Series["Series5"].LegendText = "Network Power Transfer magnitude in dB.";
            Chart1.Series["Series6"].LegendText = "Network Input Reflection Magnitude (S11) in dB.";
            Chart1.Series["Series7"].LegendText = "Network Input Reflection angle(S11), degrees";
            Chart1.Series["Series6"].LegendText = "Network Input Reflection Magnitude (S11) in dB.";
            Chart1.Series["Series7"].LegendText = "Network Input Reflection angle(S11), degrees";
            Chart1.Series["Series8"].LegendText = "";
            Chart1.Series["Series9"].LegendText = "";


            // BUGBUG
            // temporary patch to make traces more printable in black and white for magazine article
            //Chart1.Series["Series3"].BorderWidth = 2;
            //Chart1.Series["Series4"].BorderWidth = 2;
            //Chart1.Series["Series4"].BorderDashStyle = ChartDashStyle.Dash;

            // adding periodic markers to a trace (not aligned at any specific frequency)
            //Chart1.Series["Series3"].MarkerStyle = MarkerStyle.Triangle;
            //Chart1.Series["Series3"].MarkerStep = 6;
            //Chart1.Series["Series3"].MarkerSize = 6;

            // How to change the trace colors
            //Chart1.Series["Series1"].Color = Color.Black;
            //Chart1.Series["Series2"].Color = Color.Red;
            //Chart1.Series["Series3"].Color = Color.Blue;
            //Chart1.Series["Series4"].Color = Color.Green;
            //Chart1.Series["Series5"].Color = Color.Purple;
            // BUGBUG


            Chart1.Series["Series1"].XAxisType = AxisType.Primary;
            Chart1.Series["Series1"].YAxisType = AxisType.Primary;
            Chart1.Series["Series2"].XAxisType = AxisType.Primary;

            if (AnParm.InputZFormat == ComplexFormat.MagnitudePhase)
                Chart1.Series["Series2"].YAxisType = AxisType.Secondary;
            else
                Chart1.Series["Series2"].YAxisType = AxisType.Primary;

            Chart1.Series["Series3"].XAxisType = AxisType.Primary;
            Chart1.Series["Series3"].YAxisType = AxisType.Primary;
            Chart1.Series["Series4"].XAxisType = AxisType.Primary;
            Chart1.Series["Series4"].YAxisType = AxisType.Secondary;
            Chart1.Series["Series5"].XAxisType = AxisType.Primary;
            Chart1.Series["Series5"].YAxisType = AxisType.Primary;
            Chart1.Series["Series6"].XAxisType = AxisType.Primary;
            Chart1.Series["Series6"].YAxisType = AxisType.Primary;
            Chart1.Series["Series7"].XAxisType = AxisType.Primary;
            Chart1.Series["Series7"].YAxisType = AxisType.Secondary;

            // These are spares for whatever else we may want to display
            //chart1.Series["Series8"].XAxisType = AxisType.Primary;
            //chart1.Series["Series8"].YAxisType = AxisType.Primary;
            //chart1.Series["Series9"].XAxisType = AxisType.Primary;
            //chart1.Series["Series9"].YAxisType = AxisType.Secondary;


            // decide which traces to show
            Chart1.Series["Series1"].Enabled = AnParm.ShowInputZ;
            Chart1.Series["Series2"].Enabled = AnParm.ShowInputZ;
            Chart1.Series["Series3"].Enabled = AnParm.ShowVoltageTransfer;
            Chart1.Series["Series4"].Enabled = AnParm.ShowVoltageTransfer;
            Chart1.Series["Series5"].Enabled = AnParm.ShowPowerTransfer;
            Chart1.Series["Series6"].Enabled = AnParm.ShowNetworkS11;
            Chart1.Series["Series7"].Enabled = AnParm.ShowNetworkS11;
            Chart1.Series["Series8"].Enabled = false;
            Chart1.Series["Series9"].Enabled = false;

            Chart1.ChartAreas["Default"].AxisX.IsLogarithmic = (AnParm.FrequencyAxisFormat == AxisFormat.Logarithmic);
            Chart1.ChartAreas["Default"].AxisY.IsLogarithmic = (AnParm.YAxisFormat == AxisFormat.Logarithmic);
            Chart1.ChartAreas["Default"].AxisY2.IsLogarithmic = (AnParm.Y2AxisFormat == AxisFormat.Logarithmic);

            // Zoom into the X axis
            //chart1.ChartAreas["Default"].AxisX.ScaleView.Zoom(100, 1000);

            // Enable range selection and zooming end user interface
            Chart1.ChartAreas["Default"].CursorX.IsUserEnabled = true;
            Chart1.ChartAreas["Default"].CursorX.IsUserSelectionEnabled = true;
            Chart1.ChartAreas["Default"].AxisX.ScaleView.Zoomable = true;
            //chart1.ChartAreas["Default"].AxisX.ScrollBar.IsPositionedInside = true;

            // Zoom into the Y axis
            //chart1.ChartAreas["Default"].AxisY.ScaleView.Zoom(1, 10);

            // Enable range selection and zooming end user interface
            Chart1.ChartAreas["Default"].CursorY.IsUserEnabled = true;
            Chart1.ChartAreas["Default"].CursorY.IsUserSelectionEnabled = true;
            Chart1.ChartAreas["Default"].AxisY.ScaleView.Zoomable = true;
            //chart1.ChartAreas["Default"].AxisY.ScrollBar.IsPositionedInside = true;

            // change the formatting of labels on the X-Axis (frequency) to comma-separate 3-digit groups
            // http://msdn.microsoft.com/en-us/library/0c899ak8.aspx is where the format strings are defined

            Chart1.ChartAreas["Default"].AxisX.LabelStyle.Format = "#,#";


            Frequency = AnParm.StartFrequency;

            try
            {
                // iterate over the frequency sweep range
                for (int scan = 0; scan < AnParm.NumberPoints; scan++)
                {

                    // Add V=1+j0, I=0+j0 as first node V,I pair (open circuit)
                    NodeList.Add(new NodeState(Complex.One, Complex.Zero));

                    // iterate back-to-front starting at the last tile
                    for (index = TileContainer.Count - 1; index >= 0; index--)
                    {
                        Tile tile = TileContainer[index];        // get the tile
                        Boolean lastTile = (index == TileContainer.Count - 1); // if this is the last tile

                        P = tile.ExtractABCD(Frequency, lastTile);    // extract its ABCD parameters at this frequency

                        if (P.frequency == -1)
                            throw new ArgumentException("Invalid ABCD extraction");

                        NodeState N = NodeList[NodeList.Count - 1];      // get last V,I pair in the list (previous)


                        if(!tile.ForceI1I2)        // no forcing current
                            NodeList.Add(P * N);   // add [V,I] = [ABCD] * [V,I previous] to the end of the list
                        else                       // else revise V2 based on forced I1, update in and out V,I pairs 
                        {
                            Complex I1 = new Complex(tile.I1I2ratio * N.I);  // I1 = I1/I2 * I2.  Force current I1
                            N.V = (I1 - P.P22D * N.I) / P.P21C;                // New value for V2

                            Complex V1 = new Complex(P.P11A * N.V + P.P12B * N.I);  // Compute V1, we know I1
                            NodeState In = new NodeState(V1, I1);
                            NodeList.Add(In);
                        }
                    }

                    NodeList.Reverse();     // element [zero] is now before the first tile,
                    // element [count-1] is after the last tile.

                    // the nodes to analyze between
                    NodeState fromNode = NodeList[AnParm.FromNode];   // left-mode node
                    NodeState toNode = NodeList[AnParm.ToNode];  // right-most node

                    // Add InputZ of the first node to the chart series

                    if (Complex.Magnitude(fromNode.I) == 0)     // test for infinite impedance
                    {
                        Chart1.Series["Series1"].Enabled = false;
                        Chart1.Series["Series2"].Enabled = false;

                        if (AnParm.ShowInputZ)
                            TraceErrorFlag = true;
                    }
                    else                                        // input Z should be finite
                    {
                        if (AnParm.InputZFormat == ComplexFormat.MagnitudePhase)
                        {
                            Chart1.Series["Series1"].Points.AddXY(Frequency, Complex.Magnitude(fromNode.V / fromNode.I));
                            Chart1.Series["Series2"].Points.AddXY(Frequency, Complex.ArgDegr(fromNode.V / fromNode.I));
                        }
                        else    // format is real, imaginary (resistance, reactance)
                        {
                            Chart1.Series["Series1"].Points.AddXY(Frequency, (fromNode.V / fromNode.I).real);
                            Chart1.Series["Series2"].Points.AddXY(Frequency, (fromNode.V / fromNode.I).imag);
                        }
                    }

                    // Voltage Transfer function = to/from
                    // 1+j0 is the output of the last tile, V1 is the input to the network
                    // Thus transfer function is 1/input
                    Chart1.Series["Series3"].Points.AddXY(Frequency, Complex.MagnitudeDB(toNode.V / fromNode.V));
                    Chart1.Series["Series4"].Points.AddXY(Frequency, Complex.ArgDegr(toNode.V / fromNode.V));

                    // Power Transfer function = to/from.
                    // Note that the last node is open-circuit, so it dissipates no power. Normally one
                    // would select the next-to-last node to measure (the power into the last tile).
                    Double powerload = (toNode.V * Complex.Conjugate(toNode.I)).real / 2;
                    Double powersource = (fromNode.V * Complex.Conjugate(fromNode.I)).real / 2;

                    if (powerload == 0)
                    {
                        Chart1.Series["Series5"].Enabled = false;

                        if (AnParm.ShowPowerTransfer)
                            TraceErrorFlag = true;
                    }
                    else
                        Chart1.Series["Series5"].Points.AddXY(Frequency, 10 * Math.Log10(powerload / powersource));

                    // Input reflection (S11) normalized to display reference resistance
                    if (AnParm.RefResistanceS11Display == 0)
                        AnParm.RefResistanceS11Display = 50;  // use 50 ohms if no value supplied

                    if (Complex.Magnitude(fromNode.I) == 0)
                    {
                        Chart1.Series["Series6"].Enabled = false;
                        Chart1.Series["Series7"].Enabled = false;

                        if (AnParm.ShowNetworkS11)
                            TraceErrorFlag = true;
                    }
                    else
                    {
                        Complex Zn = (fromNode.V / fromNode.I) / new Complex(AnParm.RefResistanceS11Display);
                        Complex S11 = (Zn - Complex.One) / (Zn + Complex.One);

                        Double S11Loss = Complex.MagnitudeDB(S11);
                        if ((S11Loss == Double.MinValue) ||
                            (S11Loss == Double.NegativeInfinity)) S11Loss = -308; // very small loss

                        Chart1.Series["Series6"].Points.AddXY(Frequency, S11Loss);
                        Chart1.Series["Series7"].Points.AddXY(Frequency, Complex.ArgDegr(S11));
                    }


                    // increment the frequency
                    if (AnParm.FrequencyAxisFormat == AxisFormat.Linear)
                        Frequency += (AnParm.StopFrequency - AnParm.StartFrequency) / AnParm.NumberPoints;
                    else
                        Frequency *= Math.Pow(10, (Math.Log10(AnParm.StopFrequency / AnParm.StartFrequency) /
                            AnParm.NumberPoints));
                }
            }
            catch (ArgumentException)
            {
                MessageBox.Show("Cannot continue analysis. Check for: \n\rUnterminated network," +
                    "\n\rIncomplete 2-port parameter sets (zero S12 and S21, etc.)." +
                    "\n\rAn incomplete 2-port must be the last tile in the network",
                    "Fatal Analysis Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
          
            Chart1.ChartAreas["Default"].AxisX.Interval = 0;         // Auto mode
            Chart1.ChartAreas["Default"].AxisY.Interval = 0;         // Auto mode

            if (YAutocheckBox.Checked)
            {
                Chart1.ChartAreas["Default"].AxisY.Maximum = Double.NaN; // Auto mode
                Chart1.ChartAreas["Default"].AxisY.Minimum = Double.NaN; // Auto mode
            }
            else
            {
                Chart1.ChartAreas["Default"].AxisY.Maximum = AnParm.Ymax;
                Chart1.ChartAreas["Default"].AxisY.Minimum = AnParm.Ymin;
            }
            Chart1.ChartAreas["Default"].AxisY2.Minimum = -180; 
            Chart1.ChartAreas["Default"].AxisY2.Maximum = 180; 
            Chart1.ChartAreas["Default"].AxisY2.Interval = 45;


            if (XAutocheckBox.Checked)
            {
                Chart1.ChartAreas["Default"].AxisX.Maximum = Double.NaN; // Auto mode
                Chart1.ChartAreas["Default"].AxisX.Minimum = Double.NaN; // Auto mode
            }
            else
            {
                Chart1.ChartAreas["Default"].AxisX.Maximum = AnParm.StopFrequency;
                Chart1.ChartAreas["Default"].AxisX.Minimum = AnParm.StartFrequency;
            }

            Boolean showdB = AnParm.ShowVoltageTransfer || AnParm.ShowPowerTransfer ||
                AnParm.ShowNetworkS11;
            Boolean showOhms = AnParm.ShowInputZ;

            if (showdB && showOhms)
                Chart1.ChartAreas["Default"].AxisY.Title = "Ohms and dB.";
            else if (showdB)
                Chart1.ChartAreas["Default"].AxisY.Title = "dB.";
            else
                Chart1.ChartAreas["Default"].AxisY.Title = "Ohms";
            
            Chart1.ChartAreas["Default"].AxisY.TextOrientation = TextOrientation.Rotated270;
            Chart1.ChartAreas["Default"].AxisY2.Title = "Degrees";
            Chart1.ChartAreas["Default"].AxisY2.TextOrientation = TextOrientation.Rotated270;
            Chart1.ChartAreas["Default"].AxisX.Title = "Frequency, Hz.";
            Chart1.ChartAreas["Default"].AxisX.TextOrientation = TextOrientation.Horizontal;

            if (TraceErrorFlag)
            {
                MessageBox.Show("Error - possible attempt to compute Power Transfer beyond last Tile (open circuit)"+
                    "\n\rPower Transfer must be computed into the last Tile (not out of it).",
                    "Invalid Analysis Request", MessageBoxButtons.OK, MessageBoxIcon.Error);

                AnalyzeButton.BackColor = Color.Pink;
            }
            else
                AnalyzeButton.BackColor = Color.Transparent;
        }

        /// <summary>
        /// Update analysis parameters the analyze the network.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void DoAnalyze(object sender, EventArgs e)
        {

            // Update analysis parameters based on how user set the check boxes
            // and parameters for the analysis

            AnParm.ShowInputZ = InputZcheckBox.Checked;
            AnParm.InputZFormat = ReImradioButton.Checked ?
                ComplexFormat.RealImaginary : ComplexFormat.MagnitudePhase;

            AnParm.ShowNetworkS11 = S11checkBox.Checked;
            AnParm.ShowPowerTransfer = PwrTranscheckBox.Checked;
            AnParm.ShowVoltageTransfer = VTranscheckBox.Checked;

            AnParm.FrequencyAxisFormat = XLogcheckBox.Checked ?
                AxisFormat.Logarithmic : AxisFormat.Linear;

            AnParm.Yauto = YAutocheckBox.Checked;
            AnParm.Xauto = XAutocheckBox.Checked;

            if (!AnParm.Yauto)
            {
                try
                {
                    AnParm.Ymin = Convert.ToDouble(YMintextBox.Text);
                    AnParm.Ymax = Convert.ToDouble(YMaxtextBox.Text);
                }
                catch (FormatException f)
                {
                    MessageBox.Show("Invalid Ymin/Ymax numeric format", f.Message,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;         // don't do analysis until values are fixed
                }
            }

            try
            {
                AnParm.StartFrequency = Convert.ToDouble(XMintextBox.Text);
                AnParm.StopFrequency = Convert.ToDouble(XMaxtextBox.Text);
            }
            catch (FormatException f)
            {
                MessageBox.Show("Invalid Xmin/Xmax numeric format", f.Message,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;         // don't do analysis until values are fixed
            }

            if (!AnParm.Yauto && (AnParm.Ymin >= AnParm.Ymax))
            {
                MessageBox.Show("Y-minimum is >= Y-Maximum", "Range Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;         // don't do analysis until values are fixed
            }

            if (AnParm.StartFrequency >= AnParm.StopFrequency)
            {
                MessageBox.Show("X-minimum is >= X-Maximum", "Domain Error",
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;         // don't do analysis until values are fixed
            }

            // get the from and to nodes to perform the analysis against

            try
            {
                AnParm.FromNode = Convert.ToInt32(FromNodetextBox.Text);
                AnParm.ToNode = Convert.ToInt32(ToNodetextBox.Text);

            }
            catch(FormatException f)
            {
                MessageBox.Show("Invalid to/from node number format", f.Message,
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;         // don't do analysis until values are fixed
            }

            if (AnParm.FromNode < 0)                  // validate requested "from" node
            {
                AnParm.FromNode = 0;
                FromNodetextBox.Text = "0";

                MessageBox.Show("'From node' cannot be less than zero (left-most)",
                    "Invalid entry", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                return;         // don't do analysis until values are fixed
            }

            if ((AnParm.ToNode > TileContainer.Count) ||   // validate requested "to" node
                AnParm.ToNode < 1)
            {
                AnParm.ToNode = TileContainer.Count;
                ToNodetextBox.Text = AnParm.ToNode.ToString();

                StringBuilder NodeErrorMessage = new StringBuilder("'To node' cannot be greater than number of nodes (right-most)." +
                "\n\r'To node' cannot be less than one." +
                "\n\n\rResetting 'From node' = 0 and 'To node' = ");

                NodeErrorMessage.AppendFormat("{0} (after final Tile)", TileContainer.Count);

                MessageBox.Show(NodeErrorMessage.ToString(),
                "Invalid entry", MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                return;         // don't do analysis until values are fixed
            }

            Analyze();
        }

        /// <summary>
        /// Populate the numeric boxes with some existing handy values
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void RectChartSelect(object sender, TabControlEventArgs e)
        {
            if (String.Compare(FromNodetextBox.Text, "") == 0)
                FromNodetextBox.Text = "0";
            else
                FromNodetextBox.Text = AnParm.FromNode.ToString();  

            if (String.Compare(ToNodetextBox.Text, "") == 0)
                ToNodetextBox.Text = NodeList.Count.ToString();
            else
                ToNodetextBox.Text = AnParm.ToNode.ToString();

            XMintextBox.Text = AnParm.StartFrequency.ToString();
            XMaxtextBox.Text = AnParm.StopFrequency.ToString();

            YMaxtextBox.Text = AnParm.Ymax.ToString();
            YMintextBox.Text = AnParm.Ymin.ToString();

        }

        /// <summary>
        /// Print Event Handler for propertyGrid1
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="ev"></param>
        private void PrintPropertyGrid1(object sender, PrintPageEventArgs ev)
        {
            Bitmap bm = new Bitmap(propertyGrid1.Width, propertyGrid1.Height);
            propertyGrid1.DrawToBitmap(bm, propertyGrid1.ClientRectangle);
            ev.Graphics.DrawImage(bm, 15, 15);
            //ev.Graphics.DrawImage(bm, ev.MarginBounds.Left, ev.MarginBounds.Top);
        }

        /// <summary>
        /// Printe Event Handler for propertyGrid2
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="ev"></param>
        private void PrintPropertyGrid2(object sender, PrintPageEventArgs ev)
        {
            Bitmap bm = new Bitmap(propertyGrid2.Width, propertyGrid2.Height);
            propertyGrid2.DrawToBitmap(bm, propertyGrid2.ClientRectangle);
            ev.Graphics.DrawImage(bm, 15, 15);
            //ev.Graphics.DrawImage(bm, ev.MarginBounds.Left, ev.MarginBounds.Top);
        }

        /// <summary>
        /// User clicked Print ICON in the toolbar
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void PrintTab(object sender, EventArgs e)
        {
            // select Printer Device
            if (printDialog1.ShowDialog() == DialogResult.OK)
                printDocument1.PrinterSettings = printDialog1.PrinterSettings;
            else return;        // don't print anything if user failed to select a printer

                // Print the active tab
            switch (tabControl1.SelectedIndex)
            {
                case 0:     // Design Tab
                    printDocument1.PrintPage += PrintDesignTab;
                    printDocument1.Print();
                    printDocument1.PrintPage -= PrintDesignTab;
                    break;

                case 1:     // Tile Browser Tab
                    printDocument1.PrintPage += PrintPropertyGrid1;
                    printDocument1.Print();
                    printDocument1.PrintPage -= PrintPropertyGrid1;
                    break;

                case 2:     // Analysis Tab
                    printDocument1.PrintPage += PrintPropertyGrid2;
                    printDocument1.Print();
                    printDocument1.PrintPage -= PrintPropertyGrid2;
                    break;

                case 3:     // Chart Display Tab
                    Chart1.Printing.PrintDocument.PrinterSettings = printDocument1.PrinterSettings;
                    Chart1.Printing.Print(false);
                    break;

                default: break;
            }


        }

        private void DisplayHelp(object sender, EventArgs e)
        {

            // See if the help file is in our current directory

            if (!File.Exists(Environment.CurrentDirectory + "\\ABCDHelp.chm"))
                MessageBox.Show("The file:  ABCDhelp.chm is not found.\n" +
                    "Please locate this file and place it in the same directory as ABCDMatrix.exe\n\n" +
                    Environment.CurrentDirectory.ToString(), "Help File not found");
            else
                Help.ShowHelp(this, Environment.CurrentDirectory + "\\ABCDHelp.chm");

        }

    }
}
