//	SWcenter60.cpp for NJQRP DDS-60 with three changes for DDS-30 at	*#*#*#*#*#*  
//	written by Dr. Sam Green, W0PCE 
//	for the Fully Automated DDS Sweep Generator Measurement System 
/*
	This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
//	MAXIM MX7543 serial D to A Converter driver  
//	synthsizes  sawtooth sweep for oscilloscope horizontal deflection 
//  With hardware set up for bipolar operation, 
//  0x000 gives (2048/2048) x (-Vref)
//  0x800 gives 0 volts
//  0xfff gives (2047/2048) x (+Vref)
//  With Vref set by LM103 to 2.0 volts, range is from -2V to +1.999V 
//		or a resolution of about 1 mV 
//  Sweeps DDS output frequency and DAC output voltage 
//	to synthsize horizontal O'scope drive 
//  with discrete output to trigger oscilloscope sweep 
//
//	This program sweeps the DDS linearly from a lower frequency to a higher frequency while 
//	the D/A converter sweeps over a fixed range from the most negative voltage 
//	to the most positive voltage with the same number of steps.   
//	Set frequency start, end, and step number parameters in the command line.  
//	Defaults are 1 MHz center frequency, 100k Hz sweep,and 4095 steps.  
//  SWDDS.EXE [CENTER] [SWEEPWIDTH] [NUMBEROFSTEPS] 
//	with CENTER and SWEEPWIDTH in integer Hertz with no units or putctuation  
//	and an integer NUMBEROFSTEPS 4096 maximum 
//	Numeric keypad allows crude increase and decrease of CENTER frequency, end SWEEPWIDTH, 
//	and NUMBEROFSTEPS and fine adjustment of CENTER and NUMBEROFSTEPS 

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <time.h>
#include <windows.h>

void delay (int time) ;
void delaym (long msecs) ;
void write40bit2d0 (int freqword, int phaseword) ;	// DDS
void gotoxy (int x, int y) ;
void changeparameters(int cc);
void write12databits2c0 (int dacword) ;				// DAC
void ProcessStartAndStop (void);
void AdjustStepCount(void); 

float RefClock = 180000000,	// 100000000 for 100MHz clock on DDS-30		*#*#*#*#*#* 
	  mult = 0x100000000/RefClock;  

int base	= 0x378,		// parallel port register addresses
	status	= base+1,
	control	= base+2,
	centerword    = 1000000*mult,  // default 1MHz center frequency 
	halfwidthword = 50000*mult ,   // default 50kHz sweep width below and above center 
	freqstartword = centerword - halfwidthword ,	//  start of sweep
	freqendword   = centerword + halfwidthword , 	//  end of end
	phaseword = 0x01,		// 0x00 for DDS-30							*#*#*#*#*#* 
	numberofsteps = 0x1000,	// default 4096 steps 
	freqword = freqstartword,
	dacstart = 0x000,	// default -Vref start
	dacend   = 0xfff, 	// default Vref*2047/2048 end
	dacword = dacstart,
	dacstepcount = (dacend-dacstart)/numberofsteps,		// default dac increment 
	freqstepcount = (freqendword-freqstartword)/numberofsteps;	// default freq increment 

void main(int argc, char *argv[]) {
	char c; int cc; 
	system("CLS");
	freqword = centerword;  
	dacword = dacstart;  

	gotoxy(1,1);printf("Usage: %s [CenterFreq] [SweepWidth] [#steps]\n", argv[0]);
									// change 60 to 30 on DDS-30		*#*#*#*#*#*
	gotoxy(1,2);printf("Integer frequencies between 1 and 60000000 with no units or punctuation\n"); 
	puts("and #steps 4096 maximum ");

	if (argc>1) centerword = mult*atoi(argv[1]); 			// Center frequency
	if (argc>2) halfwidthword   = mult*atoi(argv[2])/2;		// Sweep width 
	freqstartword = centerword - halfwidthword; 
	freqendword   = centerword + halfwidthword;
	if(freqstartword<0)freqstartword=0; 
							// change 60 to 30 four places for DDS-30	*#*#*#*#*#*
	if (freqstartword>60000000*mult) freqstartword=60000000*mult; 
	if (freqendword>60000000*mult) freqstartword=60000000*mult;
	freqstepcount = (freqendword-freqstartword)/numberofsteps; 
	if (argc>3) {numberofsteps = atoi(argv[3]); 
		freqstepcount = (freqendword-freqstartword)/numberofsteps;	// hex integer divided by integer
	} 
	if(freqstepcount < 1) freqstepcount=1;			// could be less than 1, preventing sweep
	ProcessStartAndStop ();
	while(1){								// swept frequency
		_outp(control, 8);_outp(control, 0);// pulse LPT pin 17 high & low to trigger o'scope  
		for(freqword = freqstartword; freqword < freqendword; freqword += freqstepcount) {
			write40bit2d0 (freqword, phaseword) ;
			gotoxy(3,14);printf("\tInstantaneous Frequency = %10.2f Hz ", freqword/mult);
			dacword += dacstepcount; 
			write12databits2c0 (dacword) ;

			if(kbhit()) {(c = getch()); cc=c;changeparameters(cc);break;}
			gotoxy(3,16);printf("\tDAC step count = 0x%x\n", dacstepcount);
			gotoxy(3,18);printf("\tInstantaneous DAC Word  = 0x%03x  ", dacword);
//			delaym(1);  // milliseconds
		}	dacword = dacstart ; // end and restart DAC sweep too 
}	}
	
void delay (int time){ // short delay in fractions of a microsecond
	int i;
	for (i=0; i<time; i++) ;
}
	
void delaym (long msecs) {  // long delay in milliseconds
	clock_t now = clock();  // get current time
	clock_t then = now + CLOCKS_PER_SEC * (long)msecs/1000 ;
	while (now<then) now=clock() ;
}

void write40bit2d0 (int freqword, int phaseword) { // with clock to d1 & freq update strobe to d2
	int freqvar, i, writebit;

	freqvar = freqword ;
	for(i=1; i<33; i++) {           
		writebit = freqvar & 1 ;	
		_outp(base, writebit);		// write bit
		_outp(base, writebit | 2);	// writeclock high
		_outp(base, writebit);	// writeclock low 
		freqvar >>= 1 ;
	}
	freqvar = phaseword ;  // all zeroes 
	for(i=1; i<9; i++) {           
		writebit = freqvar & 1 ;	
		_outp(base, writebit);		// write bit
		_outp(base, writebit | 2);	// writeclock high
		_outp(base, writebit);	// writeclock low 
		freqvar >>= 1 ;
	}
	_outp(base, 4);	// frequency update stobe pulse high;  
	_outp(base, 0);	// frequency update stobe pulse low;  
}

void write12databits2c0 (int dacword) { // with clock to c1 & transfer update strobe to c2 
	int dacvar, i, writebit, mask=0x1000;

	dacvar = ~dacword ;					// bitwise invert to write data to inverted C0 
	for(i=1; i<13; i++) {           
		mask >>= 1 ; writebit = dacvar & mask ; if (writebit!=0) writebit = 1; // reverses dacvar 
		_outp(base+2, 4 + writebit);	// write bit & keep both load (4) and clear (8) inactive
		_outp(base+2, 6 + writebit);	// writeclock low through inverted C1 & keep both load and clear inactive
		_outp(base+2, 4 + writebit);	// writeclock high through inverted C1 & keep both load and clear inactive
		}
	_outp(base+2, 0);					// update stobe pulse low noninverted C2
	_outp(base+2, 4);					// update stobe pulse high  
}

void gotoxy (int x, int y){
	HANDLE hdl;
	COORD coords;
	hdl = GetStdHandle(STD_OUTPUT_HANDLE);
	coords.X=x-1;
	coords.Y=y-1;
	SetConsoleCursorPosition(hdl,coords);
}

void changeparameters(int cc){
	switch (cc){
		case 0x37:  centerword += centerword/10;										// 7
					ProcessStartAndStop();
					break;	

		case 0x31:  centerword -= centerword/10;										// 1
					ProcessStartAndStop();
					break;	

		case 0x39:  halfwidthword *= 2  ;												// 9
					ProcessStartAndStop();
					break;	

		case 0x33:  halfwidthword /= 2  ;     											// 3
					ProcessStartAndStop();
					break;	

		case 0x38:  numberofsteps/=2  ;  												// 8
					AdjustStepCount();
					break; 
		
		case 0x32:  numberofsteps*=2  ;     											// 2
					AdjustStepCount();
					break;	

		case 0x34:  centerword -= freqstepcount;										// 4
					ProcessStartAndStop();
					break;	

		case 0x36:  centerword += freqstepcount;										// 6
					ProcessStartAndStop();
					break;	

		case 0x35:  numberofsteps += numberofsteps/16;									// 5
					AdjustStepCount();
					break;	
		
		default:	;	
}	}

void ProcessStartAndStop (void){
	freqstartword = centerword - halfwidthword; 
							// change 30 to 60 six places on DDS-60	*#*#*#*#*#*
	if(freqstartword<0)freqstartword=0; 
	if (freqstartword>60000000*mult) freqstartword = 60000000*mult; 
	freqendword = centerword + halfwidthword;
	if (freqendword > 60000000*mult) {
		freqendword = 60000000*mult; // comment out this line to disable upper frequency limit 
		gotoxy(50,8);puts ("60 MHz limit!");
		}
	gotoxy(1,5);printf("DDS-60\tCenter Frequency\t= %10.1f Hz  ", (centerword/mult));
	gotoxy(3,6);printf("\tTotal Sweep Width\t= %10.1f Hz  ", (2*halfwidthword/mult));
	gotoxy(3,7);printf("\tStart Frequency\t\t= %10.1f Hz      ", (freqstartword/mult)); 
	gotoxy(3,8);printf("\tEnd Frequency\t\t= %10.1f Hz  "	  ,   (freqendword/mult));
	AdjustStepCount();
}


void AdjustStepCount (void) {
	if (numberofsteps > 4096) {numberofsteps = 4096; gotoxy(44,10); puts ("4096 step limit!");} // DAC steps  
	if (dacstepcount == 0) dacstepcount = 1;
	freqstepcount = (freqendword-freqstartword)/numberofsteps; 
	if(freqstepcount < 1) freqstepcount=1;			// could be less than 1, preventing sweep
	dacstepcount  = 1+(dacend-dacstart)/numberofsteps; 
	gotoxy(3,10);printf("\tNumber of Steps\t\t= %d   ",numberofsteps);
	gotoxy(3,11);printf("\tCount Increment\t\t= %d    \n", freqstepcount); 
	gotoxy(3,12);printf("\tFrequency Increment\t= %10.2f Hz  \n", freqstepcount/mult); 

}