/*	BERT.C -- BER test program. Operates in conjunction with
 *	TI DSK board via serial port.
 */

#include <stdio.h>
#include <stdlib.h>
#include <bios.h>
#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include <math.h>
#include <time.h>

#include "comport.h"

static COMM *comm;
static char verbose = 0;

void
gone()
{
	close_com(comm);
}

/*	r e c v _ d s p
 *
 *	Receive a character from the DSP. Return -1 after "timeout" msec
 * of waiting for a char.
 */
int
recv_dsp(unsigned timeout)
{
	int c;
	long tval = timeout;

	while ((c = com_in(comm)) == -1) {
		if (tval <= 0) {
			if (verbose)
				printf("\nTimeout\n");
			return -1;
		}
		if (bioskey(1) != 0)
			if ((bioskey(0) & 0xff) == 0x1b)
				return -1;
		delay(10);
		tval -= 10;
	}
	if (verbose) {
		putch(c);
		printf("(%02X)", c);
	}
	return c;
}

/*	c m d _ d s p
 *
 *	Interrupts the DSP and sends a new command. Procedure:
 *		Send the command
 *		Get a "Ready" response from the DSP
 *		Send the command again
 */
int
cmd_dsp(int cmd)
{
	send_com(comm, cmd, 0);
	if (recv_dsp(1000) != 'R')
		return -1;
	send_com(comm, cmd, 0);
	return 0;
}

/*	c m d _ r e s p
 *
 *	Send a command to the DSP and wait "timeout" msec for
 *	a one-byte response.
 */
int
cmd_resp(int cmd, unsigned timeout)
{
	if ((cmd_dsp(cmd)) == -1)
		return -1;
	return recv_dsp(timeout);
}

/*	c m d _ r e s p 1 6
 *
 *	Send a command to the DSP and wait "timeout" msec for
 *	a two-byte response.
 */
long
cmd_resp16(int cmd, unsigned timeout)
{
	long l, h;

	if ((l = cmd_resp(cmd, timeout)) == -1)
		return -1;
	if ((h = recv_dsp(100)) == -1)
		return -1;
	return (h << 8) | l;
}

long
cmd_resp32(int cmd, unsigned timeout)
{
	long l, h;

	if ((l = cmd_resp16(cmd, timeout)) == -1)
		return -1;
	if ((h = recv_dsp(100)) == -1)
		return -1;
	l = ((l & 0x7fff) << 16) | h;
	if ((h = recv_dsp(100)) == -1)
		return -1;
	l = l | (h << 8);
	return l;
}

long
cmd_resp64(int cmd, unsigned timeout, long *sec)
{
	long l, h;

	if ((l = cmd_resp32(cmd, timeout)) == -1)
		return -1;
	if (sec != NULL)
		*sec = l;
	if ((l = recv_dsp(100)) == -1)
		return -1;
	if ((h = recv_dsp(100)) == -1)
		return -1;
	l = (h << 8) | l;
	if ((h = recv_dsp(100)) == -1)
		return -1;
	l = ((l & 0x7fff) << 16) | h;
	if ((h = recv_dsp(100)) == -1)
		return -1;
	l = l | (h << 8);
	return l;
}

long
ber6(void)
{
	int i;
	long l, s, n;

	s = 0;
	printf("Running BER Test...");
	for (i = 0; i < 10; i++) {
		l = cmd_resp32('5', 15000);
		if (l == -1) {
			printf("\nTimeout\n");
			return -1;
		}
		l &= 0x7fffffffL;
		s += l;
		n = 100000L * (i + 1);
		printf("\n%7ld   %7ld   %7ld   (%e)", n, l, s,
			(double) s / (double) n);
	}
	putchar('\n');
	return s;
}
	
void
usage(void)
{
	printf("Usage: bert [-v]\n");
	exit(1);
}

int
main(int argc, char *argv[])
{
	int cport = 1;
	int c, kc;
	int lastc = 'I';
	long l, l1;
	double d;

	delay(1);
	for (c = 1; c < argc; c++)
		if (argv[c][0] == '-') {
			switch (toupper(argv[c][1])) {
			case 'V':
				verbose = 1;
				break;
			default:
				usage();
			}
		} else
			usage();
	if ((comm = open_com(cport, 19200, 0, 1, 8, 0, 1000)) == NULL) {
		printf("COM%d error: %s\n", cport, comm_errmsg(comm_errno));
		return 1;
	}
	atexit(gone);
	printf("Ready...\n");
	do {
		if ((kc = bioskey(1)) != 0)
			kc = bioskey(0) & 0xff;
		kc = toupper(kc);
		if ((c = com_in(comm)) != -1)
			printf("X:%d ", c);
		if (c == -1 && comm_errno != 0)
			reset_buffer(comm);
		if (kc == 'V') {
			printf("%g\n", (double)((int)cmd_resp16(kc, 2000)) * 1.62 / (double) 0x7fff);
			cmd_dsp(lastc);
		} else if (kc == 'D') {
			l = cmd_resp16(kc, 2000);
			printf("%ld, %lx\n", l, l);
			cmd_dsp(lastc);
		} else if (kc == 'S') {
			if ((l = cmd_resp64(kc, 4000, &l1)) == -1)
				printf("Timeout!\n");
			else {
/* printf("%ld (%08lX)  %ld (%08lX)\n", l, l, l1, l1); */
				d = (double) l / (double) l1;
				if (d < 1.0)
					d = 1.0;
				printf("SINAD: %g dB", (d == 0) ? 100.0 : 10*log10(d));
				d -= 1.0;
				printf("  S/N: %g dB\n", (d == 0) ? -100.0 : 10*log10(d));
			}
			cmd_dsp(lastc);
		} else if (kc == '4' || kc == '5') {
			printf("Running BER Test...");
			l = cmd_resp32(kc, (kc == '4') ? 4000 : 15000);
			if (l == -1)
				printf("\nTimeout\n");
			else {
				l &= 0x7fffffffL;
				printf("\n%7ld   (%e)\n", l,
					(double) l / ((kc == '4') ? 10000.0 : 100000.0));
			}
			cmd_dsp(lastc);
		} else if (kc == '6' || kc == 'L') {
			while (ber6() != -1 && kc == 'L')
				;
			cmd_dsp(lastc);
		} else if (kc != 0 && kc != 0x1b)
			cmd_dsp(kc);
		if (kc == 'C' || kc == 'I' || kc == 'N')
			lastc = kc;
	} while (kc != 0x1b);
	close_com(comm);
	comm = NULL;
	return 0;
}
