/*	CEVAL.C -- Operates in conjunction with the TI DSK board via
 *	a serial port. With the RESPH (high-range) or RESPL (low range)
 *	program running on the DSK, this program sends shaped pulses
 *	through the system under test and evaluates the amplitude and
 *	phase response of that system.
 *
 *	From "Measuring System Response with DSP," QEX, Feb 1995.
 *
 *	J. Bloom, KE3Z
 *
 *	Compiles under Turbo C 2.0, Borland C++ 3.1 and Borland C++ 4.0
 *
 *	Formatted for listing with tab stop = 4
 *
 *	Copyright (c) 1995, American Radio Relay League
 *	This program can be used free of charge for noncommercial use.
 *	Contact the American Radio Relay League, Inc, 225 Main Street,
 *	Newington, CT 06111 (tel: 203-666-1541 x276) for information
 *	regarding commercial use.
 */

#include <stdio.h>
#include <stdlib.h>
#include <bios.h>
#include <conio.h>
#include <ctype.h>
#include <dos.h>
#include <math.h>
#include <string.h>
#include <graphics.h>

#include "dsk.h"
#include "comport.h"
#include "graf.h"
#include "fft.h"

/* Program constraints */
#define MAXSAMP		512
#define MINSAMP		16

/* High-range constants */
#define H_FS	23041.4746544	/* Sampling rate */
#define H_PASS  	7200		/* Upper passband limit */
#define H_STOP		8400		/* Lower stopband limit */
#define H_DISPLIM	H_PASS		/* Upper display limit */
#define HI_RANGE	0
/* Low-range constants */
#define L_FS	5760.36866359	/* Sampling rate */
#define L_PASS  	1800		/* Upper passband limit */
#define L_STOP		2200		/* Lower stopband limit */
#define L_DISPLIM	L_PASS		/* Upper display limit */
#define LO_RANGE	1
/* Other constants */
#define DEFDLY		20000		/* Default sample delay */
#define rad2deg	(180.0 / M_PI)	/* Radians to degrees factor */

static COMM *comm;

double fs = H_FS;				/* Sampling frequency */
double passband = H_PASS;		/* High end of passband */
double stopband = H_STOP;		/* Low end of stopband */
double displim = H_DISPLIM;		/* High end of display */
int nsamp = 256;				/* Number of samples */
unsigned int sudly = DEFDLY;	/* Start-up delay */
char range = HI_RANGE;			/* Current range */
int calibrated = 0;				/* Calibration flag (<> 0 means calibarated) */
int averaging = 1;				/* Number of measurements to average */
enum {DC = 0, AC};				/* Coupling modes */
int coupling = AC;				/* Current coupling mode */
static char *pgmname;			/* Name of this program */
char title[BUFSIZ];				/* Graph title */

/* Arrays to hold the samples and spectra */
static int rcsamp[MAXSAMP];		/* Transmitted pulse waveform samples */
static double sampin[MAXSAMP];	/* Input samples */
static COMPLEX samples[MAXSAMP]; /* Input samples as complex numbers */
static COMPLEX f[MAXSAMP];		/* Spectrum of the input signal */
static COMPLEX calf[MAXSAMP];	/* Spectrum of the calibration signal */
static double gphdat[MAXSAMP];	/* Data for graphing */

/* Array to hold FFT butterfly coefficients */
static COMPLEX twid[MAXSAMP];

/*	g o n e
 *
 *	Closes serial I/O (required to restore interrupt system).
 *	Register as an "atexit" function.
 */
void
gone()
{
	close_com(comm);
}
	
void
usage(void)
{
	printf("Usage: %s [-c#] [-b#] [-h] [-l] [-m]\n", pgmname);
	printf("       -c#   Use COM# for I/O to DSK\n");
	printf("       -b#   Use # baud to DSK\n");
	printf("       -h    High range (default) -- RESPH.DSK must be loaded\n");
	printf("       -l    Low range -- RESPL.DSK must be loaded\n");
	printf("       -m    Monochrome graphics\n");
	exit(1);
}

/*	r c r e s p
 *
 *	Calculates the samples of the pulse to transmit. The pulse
 *	has a raised-cosine spectrum that is based on the specified
 *	passband and stopband limits. The sample array is based on
 *	the specified sampling frequency and number of samples.
 *	Note that the result can be considered to be the impulse
 *	response of an FIR filter. Running an impulse through the
 *	filter will generate the shaped pulse.
 *	(See Leon W. Couch, _Digital and Analog Communication Systems_,
 *	3rd ed., p 179.)
 */
void
rcresp(int *sary, int ntap, double fs, double p, double b)
{
	int i;
	double f0, fd, k, t, d, x;

	f0 = (p + b) / 2.0;		/* 6-dB point */
	fd = b - f0;			/* Delta f */
	for (i = 0; i < ntap; i++) {
		k = i - (ntap - 1) / 2.0;
		t = k / fs;
		if (k == 0)
			sary[i] = 0x7fff;
		else {
			d = 2 * M_PI * f0 * t;
			d = sin(d) / d;
			x = 4.0 * fd * t;
			d = d * cos(2 * M_PI * fd * t) / (1.0 - x * x);
			sary[i] = (int)(d * 0x7fff + ((d < 0) ? -0.5 : 0.5));
		}
	}
}

/*	c r a s h o u t
 *
 *	Prints a message and dies when an I/O error occurs with
 *	the DSK.
 */
void
crashout(char *cmd)
{
	fprintf(stderr, "I/O error to DSK: %s\n", cmd);
	exit(1);
}

/*	i n i t _ s y s t e m
 *
 *	Initializes the system for the current operating mode.
 *	1) Calculates the pulse samples to send.
 *	2) Downloads the samples to the DSK
 *	3) Calculates the twiddle factors for subsequent FFTs.
 */
void
init_system(void)
{
	int i;

	/* Generate the samples to transmit */

	rcresp(rcsamp, nsamp, fs, passband, stopband);
	
	/* Download the samples to the DSK */

	if (cmd_dsp(comm, 'T') != 0)
		crashout("T command");
	send_dsp_16(comm, nsamp);
	for (i = 0; i < nsamp; i++)
		send_dsp_16(comm, rcsamp[i]);

	/* Generate the FFT twiddle factors */

	gentwid(nsamp, twid, MAXSAMP);
}

/*	g e t _ s a m p l e s
 *
 *	Gets the system response by telling the DSK to take one
 *	measurement. Repeats "averaging" times and averages the
 *	measurements.
 */
void
get_samples(double *ary, int averaging)
{
	int i, j;
	long l;

	clrscr();
	for (j = 0; j < averaging; j++) {
		gotoxy(1, 25);
		cputs("Sampling data");
		if (averaging > 1)
			cprintf(" (pass %d)...", j + 1);
		else
			cputs("...");
		if (cmd_dsp(comm, 'R') != 0)
			crashout("R command");
		send_dsp_16(comm, nsamp);
		if (cmd_dsp(comm, 'S') != 0)
			crashout("S command");
		send_dsp_16(comm, sudly);
		if (cmd_dsp(comm, 'G') != 0)
			crashout("G command");
		for (i = 0; i < nsamp; i++) {
			if ((l = recv_dsp_16(comm, 6000)) == -1)
				crashout("data input");
			if (j == 0)
				ary[i] = 0;
			ary[i] += (double)((int)l) / 0x7fff;
		}
	}
	if (averaging > 1)
		for (i = 0; i < nsamp; i++)
			ary[i] /= averaging;
}

/*	m a i n _ m e n u
 *
 *	Diaplays the main menu.
 */
void
main_menu(void)
{
	char buf[BUFSIZ];

	clrscr();
	sprintf(buf, "\r\nSystem is %s\r\n",  calibrated ? "calibrated" : "UNCALIBRATED");
	cputs(buf);
	sprintf(buf, "\r\n%d samples (%s coupled) Averaging=%d\r\n\r\n", nsamp,
		(coupling == AC) ? "AC" : "DC", averaging);
	cputs(buf);
	cputs("S - Sample system\r\n");
	cputs("N - Number of samples\r\n");
	cputs("C - Calibrate\r\n");
	cputs("U - Uncalibrate\r\n");
	cputs("A - AC coupled\r\n");
	cputs("D - DC coupled\r\n");
	cputs("V - Averaging\r\n");
	cputs("I - Idle signal\r\n");
	cputs("T - Title\r\n");
	cputs("Q - Quit\r\n");
	cputs("\r\ncmd>");
}

/*	s p e c t r u m
 *
 *	Places the input samples into an array of complex numbers
 *	for the FFT and executes the FFT to calculate the spectrum.
 */
void
spectrum(COMPLEX *ary)
{
	int i;

	clrscr();
	gotoxy(1, 25);
	cputs("Running FFT...");
	for (i = 0; i < nsamp; i++) {
		samples[i].r = sampin[i];
		samples[i].i = 0;
	}
	fft(samples, nsamp, ary, MAXSAMP, twid, NULL);
	gotoxy(1, 25);
	clreol();
}

/*	c o m p l e x _ d i v i d e
 *
 *	Divides c1 by c2.
 */
COMPLEX *
complex_divide(COMPLEX *c1, COMPLEX *c2, COMPLEX *q)
{
	double r, mag;

	mag = c2->r * c2->r + c2->i * c2->i;
	r = (c1->r * c2->r + c1->i * c2->i) / mag;
	q->i = (c1->i * c2->r - c1->r * c2->i) / mag;
	q->r = r;
	return q;
}

/*	c _ s p e c t r u m
 *
 *	Adjusts the measured spectrum for calibration by dividing
 *	each frequency bin value by the corresponding value from
 *	the calibration spectrum.
 */
void
c_spectrum(void)
{
	int i;

	spectrum(f);
	if (calibrated)
		for (i = 0; i < nsamp; i++)
			complex_divide(f + i, calf + i, f + i);
}

/*	g r a p h _ g d e l a y
 *
 *	Calculates group delay and displays it on a graph.
 */
void
graph_gdelay(void)
{
	int i, end;
	double phs, oldphs, gd;
	double span, ospan, median;
	double min = 10000, max = -10000;

	end = (displim * nsamp + fs - 1) / fs;
	oldphs = atan2(f[1].i, f[1].r);
	for (i = 2; i < end+1; i++) {
		phs = atan2(f[i].i, f[i].r);
		gd = oldphs - phs;
		if (fabs(gd) > (M_PI / 2))
			gd = gd - (M_PI * 2) * ((gd < 0) ? -1 : 1);
		gd = 1000000.0 * gd / ((2 * M_PI) * fs / nsamp);
		gphdat[i-1] = gd;
		if (gd < min)
			min = gd;
		if (gd > max)
			max = gd;
		oldphs = phs;
	}
	span = (max - min) * 1.1;
	if (span < 20)
		span = 20;
	ospan = pow(10.0, (int)(log10(span)));
	span = (int)(span / ospan + 1) * ospan;
	median = (max + min) / 2.0;
	ospan = ospan / 10.0;
	median = (int)(median / ospan) * ospan;
	min = median - span / 2.0;
	max = median + span / 2.0;
	r_graph(gphdat, end, 1, 0, (end * fs) / nsamp, 10, min, max, 11,
		"Frequency", "Group Delay (usec)");
}

/*	g r a p h _ p h a s e
 *
 *	Graphs the phase response and displays it on a graph.
 */
void
graph_phase(void)
{
	int i, end;

	end = (displim * nsamp + fs - 1) / fs;
	for (i = coupling; i < end; i++)
		gphdat[i] = atan2(f[i].i, f[i].r) * rad2deg;
	r_graph(gphdat, end, coupling, 0, (end * fs) / nsamp, 10, -180, 180,
		19, "Frequency", "Phase (degrees)");
}

/*	g r a p h _ m a g
 *
 *	Graphs the amplitude response and displays it on a graph.
 *	The "db" parameter defines the scaling of the amplitude
 *	(vertical) axis of the graph. If db=0, a linear scale is used.
 *	A positive value for dB results in a dB scale, with a range
 *	equal to 5 times the value of dB.
 */
void
graph_mag(double db)
{
	int i, end;
	double amax = 0;

	end = (displim * nsamp + fs - 1) / fs;
	for (i = coupling; i < end; i++) {
		gphdat[i] = sqrt(f[i].r * f[i].r + f[i].i * f[i].i);
		if (gphdat[i] > amax)
			amax = gphdat[i];
	}
	for (i = coupling; i < end; i++) {
		gphdat[i] = gphdat[i] / amax;
		if (db != 0) {
			if (gphdat[i] == 0)
				gphdat[i] = -db * 5;
			else
				gphdat[i] = 20 * log10(gphdat[i]);
			if (gphdat[i] < -db * 5)
				gphdat[i] = -db * 5;
		}
	}
	amax = (end * fs) / nsamp;
	if (db == 0)
		r_graph(gphdat, end, coupling, 0, amax, 10, 0, 1, 11, "Frequency",
			"Amplitude");
	else
		r_graph(gphdat, end, coupling, 0, amax, 10, -db * 5, 0, 11,
			"Frequency", "Amplitude (dB)");
}

/*	g e t _ f n a m e
 *
 *	Prompts the user for a file name to save or load. Returns
 *	NULL if no file name is given.
 */
char *
get_fname(char *type)
{
	static char buf[BUFSIZ];
	char *cp;

	printf("\nFile to %s [.RSP]: ", type);
	gets(buf);
	for (cp = buf; *cp == ' ' || *cp == '\t'; cp++)
		;
	if (*cp == 0)
		return NULL;
	if (strchr(cp, '.') == NULL)
		strcat(cp, ".RSP");
	strupr(cp);
	return cp;
}

/*	s a v e _ f i l e
 *
 *	Saves the current system parameters, input data and (if
 *	calibrated) calibration array.
 */
void
save_file(void)
{
	char *fn;
	FILE *f;
	int i;

	restorecrtmode();
	clrscr();
	while (1) {
		if ((fn = get_fname("save")) == NULL)
			return;
		if ((f = fopen(fn, "w")) == NULL)
			perror(fn);
		else {
			fprintf(f, "%s\n", title);
			fprintf(f, "%d\n%d\n%d\n%u\n%d\n", nsamp, averaging, calibrated,
				sudly, coupling);
			fprintf(f, "%g\n%g\n%g\n%g\n", fs, passband, stopband, displim);
			for (i = 0; i < nsamp; i++)
				fprintf(f, "%g\n", sampin[i]);
			if (calibrated)
				for (i = 0; i < nsamp; i++)
					fprintf(f, "%g\n%g\n", calf[i].r, calf[i].i);
			fclose(f);
			return;
		}
	}
}

/*	f _ d e c o d e
 *
 *	Reads a line from the file and decodes the real number thereon.
 *	Handles the QuickBASIC number format as well as C format.
 */
double
f_decode(FILE *f)
{
	char buf[BUFSIZ];
	char *cp;

	fgets(buf, sizeof buf, f);
	if ((cp = strchr(buf, 'D')) != NULL)
		*cp = 'e';
	return atof(buf);
}

/*	l o a d _ f i l e
 *
 *	Loads the system parameters, input data and (if calibrated)
 *	calibration array.
 */
void
load_file(void)
{
	char *fn, *cp;
	FILE *f;
	int i, j;

	restorecrtmode();
	clrscr();
	while (1) {
		if ((fn = get_fname("load")) == NULL)
			return;
		if ((f = fopen(fn, "r")) == NULL)
			perror(fn);
		else {
			fgets(title, sizeof title, f);
			if ((cp = strchr(title, '\n')) != NULL)
				*cp = 0;
			fscanf(f, "%d\n%d\n%d\n%u\n%d\n", &j, &averaging, &calibrated,
				&sudly, &coupling);
			fs = f_decode(f);
			passband = f_decode(f);
			stopband = f_decode(f);
			displim = f_decode(f);
			if (j != nsamp) {
				nsamp = j;
				init_system();
			}
			for (i = 0; i < nsamp; i++)
				sampin[i] = f_decode(f);
			if (calibrated)
				for (i = 0; i < nsamp; i++) {
					calf[i].r = f_decode(f);
					calf[i].i = f_decode(f);
				}
			if (ferror(f)) {
				perror(fn);
				bioskey(0);
			}
			fclose(f);
			return;
		}
	}
}

/*	d a t a _ m e n u
 *
 *	Handles user commands affecting sampled data. This routine
 *	runs while a graph is displayed on the screen, allowing the
 *	user to select a different graph, get a new measurement or
 *	return to the main menu.
 */
void
data_menu(void)
{
	enum { NOGRAPH, DATA, DB1, DB5, DB10, LINEAR, PHASE, GD};
	int graph = DATA;
	int havfft = 0;
	int newgraph;
	int c;
	int maxy;

	get_samples(sampin, averaging);
	do {
		switch (graph) {
		case DATA:
			r_graph(sampin, nsamp, 0, 0, nsamp, 9, -1, 1, 11, "Sample",
				"Amplitude");
			break;
		case DB1:
			if (!havfft)
				c_spectrum();
			havfft = 1;
			graph_mag(2);
			break;
		case DB5:
			if (!havfft)
				c_spectrum();
			havfft = 1;
			graph_mag(10);
			break;
		case DB10:
			if (!havfft)
				c_spectrum();
			havfft = 1;
			graph_mag(20);
			break;
		case LINEAR:
			if (!havfft)
				c_spectrum();
			havfft = 1;
			graph_mag(0);
			break;
		case PHASE:
			if (!havfft)
				c_spectrum();
			havfft = 1;
			graph_phase();
			break;
		case GD:
			if (!havfft)
				c_spectrum();
			havfft = 1;
			graph_gdelay();
			break;
		}
		maxy = getmaxy();
		outtextxy(0, maxy-20,
		"S=Sample   W=Waveform   1=1-dB/div   5=5-dB/div   0=10-dB/div   I=Linear");
		outtextxy(0, maxy-10, "H=Phase   G=Group delay   V=Save   L=Load   Q=Quit");
		outtextxy(getmaxx() - textwidth("XXXXXXXXXXXXX"), maxy - 10,
			calibrated ? "calibrated" : "UNCALIBRATED");
		newgraph = 0;
		while (newgraph == 0) {
			while (bioskey(1) == 0)
				;
			c = toupper(bioskey(0) & 0xff);
			switch (c) {
			case 'S':			/* Get new samples */
				havfft = 0;
				restorecrtmode();
				get_samples(sampin, averaging);
				newgraph = DATA;
				break;
			case 'W':
				newgraph = DATA;
				break;
			case '1':
				newgraph = DB1;
				break;
			case '5':
				newgraph = DB5;
				break;
			case '0':
				newgraph = DB10;
				break;
			case 'I':
				newgraph = LINEAR;
				break;
			case 'H':
				newgraph = PHASE;
				break;
			case 'G':
				newgraph = GD;
				break;
			case 'V':
				save_file();
				newgraph = graph;
				break;
			case 'L':
				load_file();
				newgraph = graph;
				break;
			case 'Q':
			case 0x1b:
				newgraph = 1;
				break;
			}
		}
		graph = newgraph;
		restorecrtmode();
	} while (c != 'Q' && c != 0x1b);
	restorecrtmode();
}

/*	s e t _ n s a m p
 *
 *	Prompts the user for the nu*mber of samples to use.
 *	Forces the number of samples to be 2^N with 4<=N<=10
 */
void
set_nsamp(void)
{
	char buf[BUFSIZ];
	int n, M;

	do {
		clrscr();
		printf("Number of samples (%d): ", nsamp);
		gets(buf);
		if ((n = atoi(buf)) == 0)
			return;
		if (n >= 16) {
			for (M = 4; M < 10; M++) {
				nsamp = 1 << M;
				if (n <= nsamp)
					break;
			}
		}
	} while (n < 16);
}

/*	c a l i b r a t e
 *
 *	Performs calibration function. Takes one measurement to
 *	determine where the maximum pulse amplitude is relative to
 *	the center of the sample array, then adjusts the start-up
 *	delay so that in succeeding measurements the pulse will
 *	be in the middle of the array. Then takes an averaged
 *	measurement, calculates the spectrum and places it in the
 *	"calf" array. Sets the "calibrated" flag.
 */
void
calibrate(void)
{
	int i, maxi = 0;
	double amp = 0;

	sudly = DEFDLY;
	get_samples(sampin, 1);
	for (i = 0; i < nsamp; i++)
		if (fabs(sampin[i]) > amp) {
			amp = fabs(sampin[i]);
			maxi = i;
		}
	sudly = sudly + maxi - nsamp / 2;
	get_samples(sampin, averaging);
	spectrum(calf);
	calibrated = 1;
}

/*	m a i n
 *
 *	1) Interprets command-line arguments
 *	2) Initializes the serial I/O to the DSK
 *	3) Loops on the main menu, accepting user commands.
 */
int
main(int argc, char *argv[])
{
	int cport = 1;
	unsigned int baud = 19200;
	int c;
	int i;
	char buf[BUFSIZ];
	char *cp;
	
	delay(1);
	if ((cp = strrchr(argv[0], '\\')) != NULL) {
		pgmname = cp + 1;
		if ((cp = strchr(cp, '.')) != NULL)
			*cp = 0;
	} else
		pgmname = argv[0];
	strlwr(pgmname);
	for (c = 1; c < argc; c++)
		if (argv[c][0] == '-') {
			switch (toupper(argv[c][1])) {
			case 'C':
				cport = atoi(argv[c]+2);
				if (cport < 1 || cport > 4)
					usage();
				break;
			case 'B':
				baud = atoi(argv[c]+2);
				if (baud == 0)
					usage();
				break;
			case 'H':
				fs = H_FS;
				passband = H_PASS;
				stopband = H_STOP;
				displim = H_DISPLIM;
				range = HI_RANGE;
				break;
			case 'L':
				fs = L_FS;
				passband = L_PASS;
				stopband = L_STOP;
				displim = L_DISPLIM;
				range = LO_RANGE;
				break;
			case 'M':
				usecolor = 0;
				break;
			default:
				usage();
			}
		} else
			usage();

	/* Initialize the COM port to talk to the DSK */

	if ((comm = open_com(cport, baud, 0, 1, 8, 0, 1024)) == NULL) {
		printf("COM%d error: %s\n", cport, comm_errmsg(comm_errno));
		return 1;
	}
	atexit(gone);

	/* Initialize the system for the number of samples in use */

	init_system();

	/* Command loop */
	main_menu();
	do {
		while (bioskey(1) == 0)
			;
		c = toupper(bioskey(0) & 0xff);
		switch (c) {
		case 'S':		/* Sample the system */
			data_menu();
			main_menu();
			break;
		case 'N':
			i = nsamp;
			set_nsamp();
			if (i != nsamp) {
				init_system();
				calibrated = 0;
			}
			main_menu();
			break;
		case 'I':
			clrscr();
			if (cmd_dsp(comm, 'R') != 0)
				crashout("R command");
			send_dsp_16(comm, 0);
			if (cmd_dsp(comm, 'G') != 0)
				crashout("G command");
			gotoxy(1, 25);
			cputs("Hit any key to continue...");
			bioskey(0);
			cmd_dsp(comm, 'Q');
			main_menu();
			break;
		case 'V':
			clrscr();
			cprintf("Averging (%d): ", averaging);
			gets(buf);
			i = atoi(buf);
			if (i > 0 && i != averaging) {
				averaging = i;
				calibrated = 0;
			}
			main_menu();
			break;
		case 'C':
			calibrate();
			main_menu();
			break;
		case 'A':
			coupling = AC;
			main_menu();
			break;
		case 'D':
			coupling = DC;
			main_menu();
			break;
		case 'U':
			calibrated = 0;
			main_menu();
			break;
		case 'T':
			clrscr();
			cputs("Plot title: ");
			gets(title);
			main_menu();
			break;
		}
	} while (c != 'Q');
	clrscr();
	close_com(comm);
	comm = NULL;
	return 0;
}
