/*	Fast Fourier transform
 *
 *	J. Bloom
 *	1/1/95
 *
 *	Computes the FFT using an in-place decimation-in-time algorithm.
 *
 *	Copyright (c) 1995, American Radio Relay League
 *	This program can be used free of charge for noncommercial use.
 *	Contact the American Radio Relay League, Inc, 225 Main Street,
 *	Newington, CT 06111 (tel: 203-666-1541 x276) for information
 *	regarding commercial use.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <io.h>
#include <float.h>

#include "fft.h"

/*	b i t r e v
 *
 *	Reverse the order of the bits in an integer.
 */
static int
bitrev(int idx, int n)
{
	int i, newv = 0;

	for (i = 0; i < n / 2; i++)
		newv = newv | ((idx & (1 << i)) << (n - 1 - 2 * i));
	for (i = n / 2; i < n; i++)
		newv = newv | ((idx & (1 << i)) >> (2 * i - (n - 1)));
	return newv;
}

/*	c o m p l e x _ m u l t
 *
 *	Multiply two complex numbers
 */
COMPLEX *
complex_mult(COMPLEX *m1, COMPLEX *m2, COMPLEX *p)
{
	double Re;

	Re = m1->r * m2->r - m1->i * m2->i;
	p->i = m1->r * m2->i + m1->i * m2->r;
	p->r = Re;
	return p;
}

/* b u t t e r f l y
 *
 * Computes the standard radix-2 butterfly
*/
static void
butterfly(COMPLEX *a1, COMPLEX *a2, COMPLEX *wnp)
{
	COMPLEX p;

	complex_mult(a2, wnp, &p);
	a2->r = a1->r - p.r;
	a2->i = a1->i - p.i;
	a1->r = a1->r + p.r;
	a1->i = a1->i + p.i;
}

/*	g e n t w i d
 *
 *	Calculate the twiddle factors, returning them in an alloc'd array
 */
COMPLEX *
gentwid(int nsamp, COMPLEX *twiddles, int ntwid)
{
	int M, N;
	int i, m, n;
	double d, dd;
	int ERF;
	COMPLEX *tp;

	/* Determine size of needed FFT (>= # of input samples) */

	for (M = 0; M < 14; M++) {
		N = 1 << M;
		if (nsamp <= N)
			break;
	}
	if (twiddles != NULL && ntwid < N)
		return NULL;

	/* Allocate array to hold twiddle factors */

	if (twiddles == NULL)
		if ((twiddles = (COMPLEX *) malloc(N * sizeof (COMPLEX))) == NULL)
			return NULL;
	tp = twiddles;
	
	/* Calculate twiddle factors in the order needed by fft() */

	for (m = 0; m < M; m++)		/* Stage */
	{
		ERF = 1 << (M-1 - m);	/* Exponent repeat factor */
		d = 0;
		n = 1 << m;
		for (i = 0; i < n; i++)
		{
			dd = 2 * M_PI * d;
			tp->r = cos(dd);
			tp->i = -sin(dd);
			d = d + (double) ERF / (double) N;
			tp++;
		}
	}
	return twiddles;
}

/*	f f t
 *
 *	Performs an FFT
 *
 *	dat			Input sample data (complex)
 *	ndat		# of samples in dat
 *	samples		Result data array (optional)
 *	nsamp		# elements in samples
 *	twiddles	array of twiddle factor from gentwid (optional)
 *	nbin			pointer to where # bins is placed (may be NULL)
 */
COMPLEX *
fft(COMPLEX *dat, int ndat, COMPLEX *samples, int nsamp,
	COMPLEX *twiddles, int *nbin)
{
	int i, idx;
	int N, M;
	int j, lc;
	int ERF, m, n;
	COMPLEX *tp;
	char needtwid = (twiddles == NULL);
	char needsamp = (samples == NULL);

	/* Determine size of needed FFT (>= # of input samples) */

	for (M = 0; M < 14; M++) {
		N = 1 << M;
		if (ndat <= N)
			break;
	}
	if (nbin != NULL)		/* Report FFT size */
		*nbin = N;
	if (needsamp && nsamp < N)	/* Result array too small */
		return NULL;

	/* If needed, allocate an array to hold the samples and the result */

	if (needsamp)
		if ((samples = (COMPLEX *) malloc(N * sizeof (COMPLEX))) == NULL)
			return NULL;

	/* Zero-fill the array */

	memset(samples, 0, N * sizeof (COMPLEX));

	/* Copy the samples to the complex data array in bit-reversed order */

	for (i = 0; i < ndat && i < N; i++)
	{
		idx = bitrev(i, M);
		samples[idx].r = dat[i].r;
		samples[idx].i = dat[i].i;
	}

	/* Generate twiddle factors if not supplied */

	if (needtwid)
		if ((twiddles = gentwid(ndat, NULL, 0)) == NULL) {
			if (needsamp)
				free(samples);
			return NULL;
		}
	tp = twiddles;

	/* Perform the FFT */

	for (m = 0; m < M; m++)		/* Stage */
	{
		ERF = 1 << (M-1 - m);	/* Exponent repeat factor */
		n = 1 << m;

		for (idx = 0, lc = 0; lc < ERF; lc++)
		{
			for (i = 0; i < n; i++)
			{
				j = i + lc * n * 2;
				butterfly(samples+j, samples+j+n, tp+idx);
				idx++;
				if (idx >= n)
					idx = 0;
				
			}
		}
		tp = tp + n;
	}
	if (needtwid)		/* Free the twiddles array if not supplied */
		free(twiddles);
	return samples;
}
