/****************************************************************/
/*                                                              */
/* SYNTH: Hybrid UHF/microwave synthesizer control routines     */
/*        for ATmega128 or Win32                                */
/*                                                              */
/* Compilable with AVR-GCC or MSVC                              */
/*                                                              */
/* ke5fx@qsl.net                                                */
/*                                                              */
/****************************************************************/

#ifdef __AVR__
#include <iom128.h>
#endif

#include <stdlib.h>
#include <string.h>

#include "typedefs.h"      

//*****************************************************************************/
//
// Pinout of cable between Atmel AVR I/O port or PC LPT port and 
// synthesizer PCB:
//
// Synth pin #  STK-500  LPTx Port
// -----------  -------  ---------
//     1          Px0      15
//     2          Px1      9
//     3          -        -   
//     4          Px2      6
//     5          Px3      7
//     6          Px4      8
//     7          Px5      4
//     8          Px6      2
//     9          Px7      3
//     10         GND      18-25
//
// AVR ports A, B, C, or D may be used to control the synthesizer.  Do NOT 
// use a straight-through 10-position ribbon cable to connect the synthesizer 
// to an Atmel STK-500 board!  
//
//*****************************************************************************/

#ifdef __AVR__
#define assert(x) if (!(x)) printf("ERROR: Assertion failed in line %d\n",__LINE__);
#endif

//
// ADF4112 / LMX2326 PLL port bit assignments
//

#ifdef __AVR__

#define PLL_DATA    (1 << 6)   // Output 
#define PLL_CLK     (1 << 7)   // Output 
#define PLL_LATCH   (1 << 5)   // Output
#define PLL_LOCKED  (1 << 0)   // Input

#else

#define PLL_DATA    0x01
#define PLL_CLK     0x02
#define PLL_LATCH   0x04
#define PLL_LOCKED  0x08

#endif

//
// AD9854 DDS port bit assignments
//

#ifdef __AVR__

#define DDS_DATA    (1 << 2)   // Output 
#define DDS_CLK     (1 << 3)   // Output 
#define DDS_LATCH   (1 << 4)   // Output 
#define DDS_MRESET  (1 << 1)   // Output 

#else

#define DDS_DATA    0x10
#define DDS_CLK     0x20
#define DDS_LATCH   0x40
#define DDS_MRESET  0x80

#endif
                       
//
// ADF4112 / PLLatinum PLL latch labels
// 

#define PLL_LATCH_R    0x00
#define PLL_LATCH_N    0x01
#define PLL_LATCH_INIT 0x03

//
// Enums used for port assignments under Atmel environment
// 

#ifdef __AVR__

enum SYNTH_PORT
{
   SYNTH_PORT_A,
   SYNTH_PORT_B,
   SYNTH_PORT_C,
   SYNTH_PORT_D,
};

#else

typedef U32 SYNTH_PORT;

#endif

//
// Enums used for PLL chip identification
//

enum PLL_CHIPTYPE
{
   LMX2306,    // National Semiconductor LMX2306
   LMX23X6,    // National Semiconductor LMX2316, LMX2326
   ADF411X,    // Analog Devices ADF4110, ADF4111, ADF4112, ADF4113
};

//
// SYNTH class definition
//

class SYNTH
{
public:

   U16          control_port;
   PLL_CHIPTYPE chip_type;
   S64          min_output_frequency;
   S32          DDS_center_frequency;

   S32          DDS_BW;
   S32          min_comparison_frequency;
   S32          max_comparison_frequency;
   S8           DDS_min_clock_multiplier;          
   S8           DDS_max_clock_multiplier;
   S32          DDS_clock_frequency;
   S32          P;
   S32          R;
   S32          current_N;
   S32          current_CM;

   // -----------------------------------------------------------
   // Constructor
   // -----------------------------------------------------------
   
   SYNTH::SYNTH(SYNTH_PORT   _control_port, 
                PLL_CHIPTYPE _chip_type,
                S64          _min_output_frequency,
                S32          _DDS_center_frequency,
                S32          _DDS_clock_frequency,
                S8           _DDS_min_clock_multiplier,
                S8           _DDS_max_clock_multiplier)
      {
      chip_type                = _chip_type;
      min_output_frequency     = _min_output_frequency;
      DDS_center_frequency     = _DDS_center_frequency;
      DDS_clock_frequency      = _DDS_clock_frequency;
      DDS_min_clock_multiplier = _DDS_min_clock_multiplier;
      DDS_max_clock_multiplier = _DDS_max_clock_multiplier;

      //
      // Port assignments for ATmega128
      //
      // All control port bits are outputs except 0
      //

#ifdef __AVR__
      switch (_control_port)
         {
         case SYNTH_PORT_A: control_port = 0x3b; PORTA = 0; DDRA = 0xFE; break;
         case SYNTH_PORT_B: control_port = 0x38; PORTB = 0; DDRB = 0xFE; break;
         case SYNTH_PORT_C: control_port = 0x35; PORTC = 0; DDRC = 0xFE; break;
         case SYNTH_PORT_D: control_port = 0x32; PORTD = 0; DDRD = 0xFE; break;
         }
#else
      control_port = (U16) _control_port;
#endif
             
      //
      // Reference modulus = 11 for 10.7 MHz DDS frequency, 32 for 21.4 MHz
      //

      R = (DDS_center_frequency < 16000000L) ? 11 : 32;

      //
      // Calculate required tuning range of DDS reference (must pass through 
      // crystal filter without excessive attenuation)
      //

      DDS_BW = S32(S64(DDS_center_frequency) * S64(DDS_center_frequency) / 
                  (min_output_frequency * S64(R)));

      //
      // Calculate minimum and maximum comparison frequencies 
      //
      // Minimum Fcomp is used as baseline to determine PLL multiplier
      // 

      min_comparison_frequency = (DDS_center_frequency - (DDS_BW / 2)) / R;
      max_comparison_frequency = (DDS_center_frequency + (DDS_BW / 2)) / R;

      //
      // NatSemi parts have fixed prescaler divisors; AD parts have programmable
      // prescalers that can support a wider range of output frequencies
      //

      if (chip_type == LMX2306)
         {
         P = 8;
         }
      else if (chip_type == LMX23X6)
         {
         P = 32;
         }
      else  
         {
         S32 min_N = S32(min_output_frequency / S64(max_comparison_frequency)) - 1;

         P = 64;

         while (P*P-P >= min_N)
            {
            P >>= 1;
            }
         }

      //
      // Initialize PLL chip with constant R and P divisors
      // 

      if (chip_type == ADF411X)
         {
         //
         // Default charge-pump gain = 1.88 mA (see ADF4112 data sheet) 
         // Not used with National parts
         //

         S32 CP = 2;      

         PLL_write_word(24, PLL_LATCH_INIT | (1L << 4)  |   
                                             (CP << 15) |   
                                    ((log2(P)-3) << 22));
         PLL_latch_data();

         PLL_write_word(24, PLL_LATCH_R | (R << 2) |   
                                         (1L << 20));
         PLL_latch_data();
         }
      else
         {
         PLL_write_word(21, PLL_LATCH_INIT | (1L << 4)); 
         PLL_latch_data();

         PLL_write_word(21, PLL_LATCH_R | (R << 2) |  
                                         (1L << 20));
         PLL_latch_data();
         }

      //
      // Send master reset to DDS and hold it for a period of time
      // guaranteed to be >= 10 DDS clock cycles
      //
      // By default, this enables auto-updates at a rate of 1/128 of 
      // the DDS clock
      //

      PORT_write(DDS_MRESET);
      IO_wait();
      PORT_write(0);

      //
      // Send control word to initialize DDS
      //
      // Disable internal update clock -- we want all future updates to be
      // manually latched by DDS_LATCH
      //
      // Wait for at least one auto-update period (about 1 uS at 100 MHz) 
      // to make sure the command to disable auto-updates has been latched,
      // then retransmit control register initialization word followed by a 
      // manual latch update, to ensure the entire command is processed
      //
      // (This is necessary because the final command byte may not have been
      // updated properly, depending on when the last internal update clock
      // occurred)
      //

      for (S32 i=0; i < 2; i++)
         {
         DDS_write_control_word(DDS_min_clock_multiplier);
         IO_wait();
         }

      DDS_latch_data();

      //
      // Back off slightly from maximum I-DAC amplitude
      // to increase SFDR
      //

      DDS_write_word(0x08);   
      DDS_write_word(0x0f);   
      DDS_write_word(0xc0);   
      DDS_latch_data();

      //
      // Force initial programming of DDS and PLL registers
      //

      current_N  = -1;
      current_CM = -1;
      }

   // -----------------------------------------------------------
   // Utility functions
   // -----------------------------------------------------------

   static void IO_wait(void)
      {
      volatile S32 v = 10;

      while (v--)
         {
#ifdef __AVR__
         volatile S32 d = 10;

         while (d--);
#else
         _asm
            {
            mov edx,0x80
            out dx,al
            }
#endif
         }
      }

   static S32 log2(S32 val)
      {
      S8 n = -1;

      while (1)
         {
         if (!val)
            {
            break;
            }

         val >>= 1;
         n++;
         }

      return n;
      }

   inline void PORT_write(U8 val)
      {
#ifdef __AVR__
      _MMIO_BYTE(control_port) = val;
#else
      U32 port = control_port;

      _asm
         {
         mov edx,port
         mov al,val
         out dx,al
         }
#endif
      }

   void BUS_write_word(S8 bits, S32 data, U8 data_line, U8 clk_line)
      {
      //
      // Drop all signals
      //

      PORT_write(0);

      for (S8 i=bits-1; i >= 0; i--)
         {
         //
         // Shift word into input register, starting with MSB
         // After asserting each bit, toggle CLK briefly to load it
         //

         if (data & (1L << i))
            {
            PORT_write(data_line);
            PORT_write(data_line | clk_line);
            PORT_write(data_line);
            }
         else
            {
            PORT_write(0);
            PORT_write(clk_line);
            PORT_write(0);
            }
         }
      }

   void DDS_write_word(S32 data)
      {
      BUS_write_word(8, data, DDS_DATA, DDS_CLK);
      }

   void PLL_write_word(S32 bits, S32 data)
      {
      BUS_write_word(bits, data, PLL_DATA, PLL_CLK);
      }

   void PLL_latch_data(void)
      {
      //
      // Assert LE briefly to update destination register from input buffer
      //

      PORT_write(PLL_LATCH);
      PORT_write(0);
      }

   void DDS_latch_data(void)
      {
      //
      // Assert LE briefly to update destination register from input buffer
      //

      PORT_write(DDS_LATCH);
      PORT_write(0);
      }

   void DDS_write_control_word(S8 clock_multiplier)
      {
      DDS_write_word(0x07);      // Write to control register (7)

      DDS_write_word(0x14);      // Power down the comparator and Q DAC

      if (clock_multiplier != 1)
         {
         DDS_write_word(clock_multiplier);
         }
      else
         {
         DDS_write_word(0x64);   // Disable internal clock multiplier
         }

      DDS_write_word(0x00);      // Single-tone mode with external latch updates 
      DDS_write_word(0x60);      // No sinc filter, enable internal AM
      }
 
   // -----------------------------------------------------------
   // Set DDS frequency 
   // DDS frequency must be specified in fixed-point 56:8 format
   // -----------------------------------------------------------

   void DDS_set_frequency(S64 fixed_point_frequency, //)
                          S8  CM)
      {
      //
      // Set clock multiplier, if it's changed
      //

      if (CM != current_CM)
         {
         current_CM = CM;

         DDS_write_control_word(CM);
         DDS_latch_data();
         }

      //
      // Calculate phase-increment value based on DDS clock rate 
      // and desired output frequency
      //
      // Due to 64-bit arithmetic constraints, we calculate only the top 36 bits 
      // of the DDS's 48-bit phase-increment value
      //
                                                           
      S64 v = (fixed_point_frequency << 28) / (DDS_clock_frequency * CM);

      v = v << 12;

      //
      // Send command to update DDS frequency word
      // 

      DDS_write_word(0x02);

      //
      // Send 48-bit phase-increment value, MSB first
      //

      for (S8 i=40; i >= 0; i -= 8)
         {
         DDS_write_word((v >> i) & 0xff);
         }

      DDS_latch_data();
      }

   // -----------------------------------------------------------
   // Calculate optimum clock-multiplier value for output frequency
   // -----------------------------------------------------------

   S8 DDS_calc_multiplier(U64  fOUT, //)
                          U32 *dist=NULL)
      {
      //
      // If using a fixed clock multiplier, return it immediately
      //

      if (DDS_min_clock_multiplier == DDS_max_clock_multiplier)
         {
         return DDS_min_clock_multiplier;
         }

      //
      // Try all DDS multipliers from DDS_min_clock_multiplier to DDS_max_clock_multiplier,
      // looking for the one that will maximize the distance between the
      // desired output frequency and the closest harmonic of DDS_CLK/2 and 
      // DDS_CLK/3
      //

      U32 best_distance   = 0;
      U32 best_multiplier = 0;

      for (S8 m=DDS_min_clock_multiplier; m <= DDS_max_clock_multiplier; m++)
         {
         U32 fCLK = m * DDS_clock_frequency;

         U32 harmonic_distance = (U32) -1;

         for (U32 d=2; d <= 3; d++)
            {
            U64 fCLK_frac = fCLK / d;

            U64 harmonic = fCLK_frac * ((fOUT + (fCLK_frac >> 1)) / fCLK_frac);

            U32 distance;

            if (harmonic > fOUT)
               {
               distance = harmonic - fOUT;
               }
            else
               {
               distance = fOUT - harmonic;
               }

            if (distance < harmonic_distance)
               {
               harmonic_distance = distance;
               }
            }

         if (harmonic_distance >= best_distance)
            {
            best_distance = harmonic_distance;
            best_multiplier = m;
            }
         }

      if (dist != NULL)
         {
         *dist = best_distance;
         }

      return best_multiplier;
      }

   // -----------------------------------------------------------
   // Set desired output frequency in Hz
   // -----------------------------------------------------------

   void set_frequency(U64 hertz)
      {
      //
      // Calculate DDS frequency and the N factor needed to multiply it to
      // the target frequency
      //

      S32 N = S32(hertz / min_comparison_frequency);

      S64 D = ((hertz * S64(R)) << 8) / N;

      //
      // Calculate modulus values for A and B counters based on N
      //
      // Performance-critical applications may omit the A and B calculations
      // for the National chips
      //

      S32 B = N / P;
      S32 A = N - (B * P);

      //
      // Make sure they're within the limits of the PLL chip
      //

      assert(N >= (P*P)-P);
      assert(B >= A);
      assert(B <= 8191);     
      assert(A >= 0);        

      if (chip_type == ADF411X)
         {
         assert(B >= 3);
         assert(A <= 63);       
         }
      else if (chip_type == LMX23X6)
         {
         assert(A <= 31);       
         }
      else if (chip_type == LMX2306)
         {
         assert(A <= 7);
         }

      //
      // Program the PLL only if N divisor has changed
      //
      // Use high-precision digital lock detection for both chip types
      //

      if (N != current_N)
         {
         current_N = N;

         if (chip_type == ADF411X)
            {
            PLL_write_word(24, PLL_LATCH_N | (A << 2) | 
                                             (B << 8));
            PLL_latch_data();
            }
         else
            {
            PLL_write_word(21, PLL_LATCH_N | (N << 2) | 
                                            (1L << 20));
            PLL_latch_data();
            }
         }

      //
      // Calculate optimal DDS clock multiplier
      //

      S8 CM = DDS_calc_multiplier(hertz);

      //
      // Set DDS frequency
      //

#if 0
      printf("\nDDS freq is %ld for %ld Hz: N=%ld, P=%ld, B=%ld, A=%ld CM=%d\n",
         S32(D / 256LL),S32(hertz),N,P,B,A,(S16) CM);
#endif

      DDS_set_frequency(D, CM);
      }

   // -----------------------------------------------------------
   // Return TRUE if PLL locked
   // -----------------------------------------------------------

   BOOL locked(void)
      {
#ifdef __AVR__
      return _MMIO_BYTE(control_port - 2) & PLL_LOCKED;
#else
      U32 result = 0;
      U32 port = control_port + 1;

      _asm
         {
         mov edx,port
         xor eax,eax
         in al,dx
         mov result,eax
         }

      return result & PLL_LOCKED;
#endif
      }
};

