from myhdl import *
from divideBy10 import divideBy10
from config import *

def adjustCountP(clk, reset, start, done, count, modeSw, intervalSw, adjCount,
					ud_start, ud_done, ud_dividend, ud_quotient, ud_initDiv):
	"""

	Adjust the value of the count according to the mode and interval

		modeSw		intervalSw		Adjustment
			0 (freq)	0 (0.1 sec)		count * 10
						1 (1.0)			count
						2 (10.0)		count / 10

			1 (period)	0 (1 cycle) 	count / 10	-	Convert to microseconds
						1 (10)			count / 100
						2 (100)			count / 1000

		The adjusted value returned in adjCount
	"""
	State = enum('STATE1', 'STATE2', 'STATE3', 'STATE4', 'STATE5',
				'STATE6', 'STATE7', 'STATE8', 'IDLE')
	state = Signal(State.IDLE)
	roundVals = tuple([5, 50, 500])
	round = Signal(intbv(0)[9:])
	k = Signal(intbv(0)[2:])

	@always(clk.negedge, reset.negedge)
	def logic():
		if reset == ACTIVE_LOW:
			done.next = INACTIVE_HIGH
			state.next = State.IDLE
		elif start == ACTIVE_LOW:
			done.next = INACTIVE_HIGH
			state.next = State.STATE1
		else:
			if state == State.STATE1:
				if modeSw == FREQUENCY_MODE:
					state.next = State.STATE2
				else:
					state.next = State.STATE5

			elif state == State.STATE2:
				# frequency mode
				if intervalSw == 0:
					# Interval is 0.1 second - multiply count by 10
					adjCount.next  = (count << 3) + (count << 1)
					done.next = ACTIVE_LOW
					state.next = State.IDLE
				elif intervalSw == 1:
					# frequency measurement interval is 1 second so no adjustment necessary
					adjCount.next = count
					done.next = ACTIVE_LOW
					state.next = State.IDLE
				elif intervalSw == 2:
					# Interval is 10 seconds - divide count by 10
					ud_initDiv.next = count + 5		# round up
					ud_start.next = ACTIVE_LOW
					state.next = State.STATE3

			elif state == State.STATE3:
				ud_start.next = INACTIVE_HIGH
				state.next = State.STATE4

			elif state == State.STATE4:
				if ud_done == ACTIVE_LOW:
					adjCount.next = ud_quotient
					done.next = ACTIVE_LOW
					state.next = State.IDLE

			elif state == State.STATE5:
				# Period measurement
				# The master clock frequency is 10 MHz so that the count in this mode has units 0.1 uSec
				# Convert to microseconds by dividing by 10, 100 or 1000 for intervals 0, 1, 2 (1, 10 or 100 cycles)
				# This is done by dividing by 10 1, 2 or 3 times.
				round.next = roundVals[intervalSw]
				k.next = intervalSw
				state.next = State.STATE6

			elif state == State.STATE6:
				ud_initDiv.next = count + round
				ud_start.next = ACTIVE_LOW
				state.next = State.STATE7

			elif state == State.STATE7:
				ud_start.next = INACTIVE_HIGH
				state.next = State.STATE8

			elif state == State.STATE8:
				if ud_done == ACTIVE_LOW:
					if k == 0:
						adjCount.next = ud_quotient
						done.next = ACTIVE_LOW
						state.next = State.IDLE
					else:
						ud_initDiv.next = ud_quotient
						k.next = k - 1
						ud_start.next = ACTIVE_LOW
						state.next = State.STATE7

			elif state == State.IDLE:
				pass
			else:
				state.next = State.IDLE
	return logic

def adjustCount(clk, reset, start, done, count, modeSw, intervalSw, adjCount):
	ud_dividend, ud_quotient, ud_initDiv = [Signal(intbv(0)[COUNT_WIDTH:]) for k in range(3)]
	ud_start, ud_done = [Signal(bool(0)) for k in range(2)]

	DIV10 = divideBy10(clk, reset, ud_start, ud_done, ud_dividend, ud_quotient, ud_initDiv)

	ACP = adjustCountP(clk, reset, start, done, count, modeSw, intervalSw, adjCount,
					ud_start, ud_done, ud_dividend, ud_quotient, ud_initDiv)

	return instances()

def testBench():

	clk, reset, start, done, modeSw = [Signal(bool(INACTIVE_HIGH)) for k in range(5)]
	count, adjCount = [Signal(intbv(0)[32:]) for k in range(2)]
	intervalSw = Signal(intbv(0)[2:])

	AC = adjustCount(clk, reset, start, done, count, modeSw, intervalSw, adjCount)


	HALF_PERIOD = delay(1)
	@always(HALF_PERIOD)
	def clockGen():
		clk.next = not clk

	@instance
	def testAdjustCount():
		reset.next = ACTIVE_LOW
		yield clk.negedge
		reset.next = INACTIVE_HIGH
		yield clk.negedge
		c = 99965432
		# mode, interval, expected value
		fcases = ((0, 0, c*10), (0, 1, c), (0, 2, (c+5)/10),
					(1, 0, (c+5)/10), (1, 1, (c+50)/100), (1, 2, (c+500)/1000))
		for mode, interval, ev in fcases:
			modeSw.next = mode
			intervalSw.next = interval
			count.next = c
			start.next = ACTIVE_LOW
			yield clk.negedge
			start.next = INACTIVE_HIGH
			yield clk.negedge
			while done == INACTIVE_HIGH:
				yield clk.negedge
			if adjCount != ev:
				print 'mode: %d  interval: %d  ev: %7d  av: %7d' % (modeSw, intervalSw, ev, adjCount)
		print 'done', now()
		raise StopSimulation

	return instances()

def test1():
	tb = testBench()
	Simulation(tb).run(5000)

def test2():
	clk, reset, start, done, modeSw = [Signal(bool(INACTIVE_HIGH)) for k in range(5)]
	count, adjCount = [Signal(intbv(0)[32:]) for k in range(2)]
	intervalSw = Signal(intbv(0)[2:])

	toVerilog(adjustCount, clk, reset, start, done, count, modeSw, intervalSw, adjCount)

if __name__ == '__main__':
	test1()

