// DCOCLK processing


#if defined(__GNUC__)  &&  defined(__MSP430__)
	#include <legacymsp430.h>
	#include "..\uart\dco.h"
	#if DCODEBUG > 0
		#include "..\utils\bin2hexS.h"
		#include "..\utils\b2d.h"
	#endif
#else
	#include "io430.h"
	#include "uart.h"
	#include "dco.h"
	#if DCODEBUG > 0
		#include "bin2hexS.h"
		#include "b2d.h"
	#endif
#endif
#include <string.h>
#include <stdlib.h>



/*---------------------------------------------------------------------------------------------------
Thus the DCO frequency is monitored periodically.
The frequency is determined by counting the number of SMCLK ticks that occur between successive
Timer A events (rising edges of ACLK). The frequency of the system clock, SMCLK, is initialized
to the nominal target frequency of, say, 2 MHz. So there will be typically 2e6/4096 = 488 SMCLK ticks
in one ACLK period.
*/


#define DCON 16			// The number of Compare samples to average
#define NDIV		(4096 / DCON)
#define TARGETSUM	((TARGET_FREQ + (NDIV/2))/NDIV)
#define SUMDELTA 	(TARGETSUM / 400)				// +-0.25 %

#define MODxMASK	0x1f
#define RSELxMASK	7
#define	DCOxMASK	0xe0

static struct {
	unsigned int casum;
	int firstSample;
	int count;
	int dif;
	unsigned int oldsum;
	unsigned int state;
	unsigned int oldcapture;
} update;

#if DCODEBUG > 0
void displayDCOFrequency(char *msg)
{
//unsigned long freq;

	//freq = ((unsigned long)update.oldsum) * NDIV;
	//bin2dec((char*)&freq, 4, USIGNED, msg, APPENDSP);
	bin2dec((char*)&update.dif, 2, SIGNED, msg, APPENDSP);
	bin2hexS((char*)&DCOCTL, 1, &msg[strlen(msg)], APPENDSP);
	bin2hexS((char*)&BCSCTL1, 1, &msg[strlen(msg)], APPENDCRLF);
	sendMsg(msg, 0);
}
#endif

static void increaseFreq(void)
{
	unsigned int dco = DCOCTL >> 5;
	if (((DCOCTL & MODxMASK) < MODxMASK) & (dco < 7))
		DCOCTL++;				// bump the modulator control value
	else
	{
		if (dco < 7)
		{
			// Increase the frequency select value and set the MODx bits to zero
			DCOCTL = (dco+1) << 5;
		}
		else
		{
			if ((BCSCTL1 & RSELxMASK) < RSELxMASK)
			{
				DCOCTL = 0;			// Set DCOx and MODx to zero
				BCSCTL1++;			// increase RSELx
			}
			else
			{
				update.state = DCO_MAXEDOUT;
				CCTL2 &= ~CCIE;			// Disable capture interrupts
			}
		}
	}
}

static void decreaseFreq(void)
{
	if ((DCOCTL & MODxMASK) > 0)
		DCOCTL--;
	else
	{
		unsigned int dco = DCOCTL >> 5;
		if (dco > 0)
		{
			// Decrement DCOx and set MODx to max value
			DCOCTL = ((dco-1) << 5) | MODxMASK;
		}
		else
		{
			if ((BCSCTL1 & RSELxMASK) > 0)
			{
				BCSCTL1--;		// Decrement RSELx
				DCOCTL = 0xff;	// and set DCOx and MODx to max values
			}
			else
			{
				update.state = DCO_MINEDOUT;
				CCTL2 &= ~CCIE;			// Disable capture interrupts
			}
		}
	}
}

void processDCOUpdate()
{	// Invoked from Timer A interrupt handler when leading edge of ACLK is detected.
unsigned int ccr2, del;
int dif;

	if ((update.state & DCO_ACTIVE) == 0)
		return;
	ccr2 = CCR2;
	del = ccr2 - update.oldcapture;
	update.oldcapture = ccr2;
	if (update.firstSample)
	{
		update.firstSample = 0;		// Skip the first capture
		return;
	}
	update.casum += del;
	update.count++;
	if (update.count == DCON)
	{
		update.oldsum = update.casum;
		update.dif = dif = update.casum - TARGETSUM;
		if (abs(dif) < SUMDELTA)
		{
			update.state = DCO_STABLE;	// Done
			CCTL2 &= ~CCIE;			// Disable capture interrupts
			return;
		}
		else if (dif < 0)
			increaseFreq();
		else
			decreaseFreq();
		update.count = update.casum = 0;
		update.state &= ~DCO_STABLE;
	}
}

void startDCOUpdate(void)
{	// Invoked by watchdog timer interrupt.
	if (update.state & DCO_ACTIVE)
		return;
	update.state = DCO_ACTIVE;
	update.firstSample = 1;
	update.count = update.casum = 0;
	CCTL2 |= CCIE;				// Enable capture interrupts
}

int dcoStatus(void)
{
	return update.state;
}
