#!/usr/bin/python3
#
#  Usage: make_trans_cal.py b u n
#       where 
#       b is the bottom frequency,
#       u is the upper frequency, and,
#       n is the number of datapoint in the scan
#
#   the output file is temp_cal.txt
#
#
# Jim Koehler, May, 2019
#

import sys
import serial
import time
import math

commPort = 'COMx'

###########################
# measure function
#
#  measure(f, comm)
#
# returns the ratio and phase
# from one measurements at 'f' Hz
# using 'comm' as the communication
# port
#
# the return value is a list with three elements:
#       the frequency
#       the ratio of secondary port amplitude to primary port amplitude, and,
#       the phase difference, primary minus secondary
#
###########################
def measure(f, comm):
    comm.write(s2b("f " + str(f) + "\r"))   # set the frequency
    d =[]
    comm.write(s2b("m\r"))
    d = str.split(b2s(comm.readline())) # the seven list elements are all strings
    f = int(d[0])

    ratio = float(d[5]) / float(d[1])
    phase = float(d[2]) - float(d[6])
    if (phase > 3.14159265):
        phase = phase - 6.2818531
    if (phase < -3.14159265):
        phase = phase + 6.2818531
    return [f, ratio, phase]      # this is a list of an integer and 2 floats

###########################
# do_scan function
#
#  do_scan(b, u, n, filename, m)
#
# the writes results to a file named 'filename'
# each line in the result file is formatted:
#       freq ratio phase ratio phase
# where the first ratio and phase are for the
# primary port and the second are for the auxiliary port
#
###########################
def do_scan(b, u, n):   # these arguments are all strings
    
    comm = serial.Serial(commPort, baudrate = 115200, timeout = 0.33)
    comm.readlines()    # flush the read buffer
    comm.write(s2b("p\n")) # an arbitrary command to clear the USB port
    comm.readlines()    # flush the read buffer

    comm.write(s2b("b " + b + "\r")) # set z_meter start frequency
    time.sleep(0.2)
    comm.write(s2b("u " + u + "\r")) # set z_meter end frequency
    time.sleep(0.2)
    comm.write(s2b("n " + n + "\r")) # set z_meter number of data points in the scan
    time.sleep(0.2)
    comm.write(s2b("l\r")) # set z_meter scan to linear
    time.sleep(0.2)
    comm.readlines()

    fh=open("temp_cal.txt", 'wb')

    for i in range(int(n)):
        interval = (int(u) - int(b)) / (int(n) - 1)
        f = int(b) + i * interval
#        print(f)
        data=measure(f, comm)
        fh.write(s2b(str(data[0]) + ' ' + str(data[1]) + ' ' + str(data[2]) + "\n"))
    fh.close()

def b2s(message):
    '''Byte to string'''
    return bytes.decode(message)

def s2b(message):
    '''string to bytes'''
    return bytearray(message, "ascii")

def main():
    global commPort
#    commPort = %Z_METER_COM%
    b = sys.argv[1]
    u = sys.argv[2]
    n = sys.argv[3]
    commPort = sys.argv[4]
    print(commPort)
    do_scan(b, u, n)

if __name__ == '__main__':
    main()
