// =====================================================================================================================================================================================================
// (c) 2021 Lynn Hansen, KU7Q															                                                                                                               |
// This Source Code Form is subject to the terms of the GNU GENERAL PUBLIC LICENSE, Version 3, 29 June 2007. A copy of this license can be found here: https://choosealicense.com/licenses/gpl-3.0/|
// =====================================================================================================================================================================================================



// THIS MODULE CONTAINS ALL THE CODE TO INTERFACE TO FLEX RADIOS


void Flex_Initialize()
{
	/*
	NOTE:
	The Flex radio only updates the transmit rfpower and tunepower when it actually transmits
	To set the power or tune power you have to enter the new power levels then transmit (or start the atu) otherwise they will reset to the old levels on the next transmit subscription update
	*/

	//we're starting (or restarting) the flex comm link - list of active slices
	//If no active slices, create one
	//Subscribe to slice and Tx
	//CTR2 only works with Slice 0


	static bool alreadyHr = false;

	static uint32_t flexStartTm = millis();

	if (alreadyHr || !gWiFi || gFlexCon == false)
	{
		return;
	}
	alreadyHr = true;

	//uint8_t tryCtr = 0; //try 4 times to connect then fail
	char adrs[20];
	gFlexIP.trim(); //get rid of spaces
	gFlexIP.toCharArray(adrs, gFlexIP.length() + 1);
	const char* fAdrs = adrs; //must be a const

	//track online time
	if (flexStartTm > 0)
	{
		SerialOut("*** Alive time (sec)= " + String((millis() - flexStartTm) / 1000), true);
		flexStartTm = millis();
	}

	flexAPI = server.available();
	flexAPI.setTimeout(2000);

	if (!flexAPI.connected())
	{
		Tx2HMI("Home.bRx.txt=`Connecting to Flex at " + gFlexIP + " `");
		flexAPI.connect(fAdrs, 4992);
		int ctr = 0;
		//wait up to 6 seconds
		while (ctr < 60)
		{
			if (!flexAPI.connected())
			{
				ctr++;
				delay(100);
			}
			else
			{
				break;
			}
		}

		SerialOut("Flex connected= " + String(flexAPI.connected()), true);

		if (flexAPI.connected() == false)
		{
			SerialOut("Unable to connect to Flex - exiting Flex_Initialization", true);
			gFlexCon = false;
			alreadyHr = false;
			return;
		}
	}
	String cmd = "";
	String reply = "";
	gFlexCmdNum = 1; //restart counter for this session	 
	reply = Tx2ESP(ESP_FLEX, "|ping"); //get initial connect dump	
	reply = Tx2ESP(ESP_FLEX, "|slice list");
	reply = Tx2ESP(ESP_FLEX, "|slice s 0 active=1 tx=1 txant=ANT" + String(gFlexAnt));
	//subscibe to slice, tx, and atu data	
	reply = Tx2ESP(ESP_FLEX, "|sub slice 0");
	reply = Tx2ESP(ESP_FLEX, "|sub tx all");
	reply = Tx2ESP(ESP_FLEX, "|sub atu all");
	gFlexPOPrev = 0; //force update on UpdateHMIDisplay()
	gFlexTOPrev = 0;
	gFlexLOPrev = 0;
	//set DSP filters to values saved in HMI eeprom - radio doesn't report filter settings on startup??
	if (gFlexWNB & 0x80)
	{
		reply = Tx2ESP(ESP_FLEX, "|slice s 0 wnb=1 wnb_level=" + String(gFlexWNB & 0x7f));
	}
	else
	{
		reply = Tx2ESP(ESP_FLEX, "|slice s 0 wnb=0");
	}
	if (gFlexNB & 0x80)
	{
		reply = Tx2ESP(ESP_FLEX, "|slice s 0 nb=1 nb_level=" + String(gFlexNB & 0x7f));
	}
	else
	{
		reply = Tx2ESP(ESP_FLEX, "|slice s 0 nb=0");
	}
	if (gFlexNR & 0x80)
	{
		reply = Tx2ESP(ESP_FLEX, "|slice s 0 nr=1 nr_level=" + String(gFlexNR & 0x7f));
	}
	else
	{
		reply = Tx2ESP(ESP_FLEX, "|slice s 0 nr=0");
	}
	if (gFlexAF & 0x80)
	{
		if (gRMode < 2)
		{
			//peaking for CW
			reply = Tx2ESP(ESP_FLEX, "|slice s 0 apf=1 apf_level=" + String(gFlexAF & 0x7f));
		}
		else
		{
			//notch for all others
			reply = Tx2ESP(ESP_FLEX, "|slice s 0 apf=1 anf_level=" + String(gFlexAF & 0x7f));
		}
	}
	else
	{
		reply = Tx2ESP(ESP_FLEX, "|slice s 0 apf=0");
	}

	alreadyHr = false;
}


String Flex_CalcFreq(uint32_t freq)
{
	//convert freq to floating for Flex command - can't do it directly because of float rounding errors
	uint32_t frq = freq;
	uint32_t val = 0;
	String f = "";
	if (frq > 99999999)
	{
		val = frq / 100000000;
		f = String(val);
		frq -= val * 100000000;
	}
	if (frq > 9999999)
	{
		val = frq / 10000000;
		f += String(val);
		frq -= val * 10000000;
	}
	else if (freq > 99999999)
	{
		f += "0";
	}
	if (frq > 999999)
	{
		val = frq / 1000000;
		f += String(val);
		frq -= val * 1000000;
	}
	else if (freq > 9999999)
	{
		f += "0";
	}
	f += "."; //add the decimal point
	if (frq > 99999)
	{
		val = frq / 100000;
		f += String(val);
		frq -= val * 100000;
	}
	else
	{
		f += "0";
	}
	if (frq > 9999)
	{
		val = frq / 10000;
		f += String(val);
		frq -= val * 10000;
	}
	else
	{
		f += "0";
	}
	if (frq > 999)
	{
		val = frq / 1000;
		f += String(val);
		frq -= val * 1000;
	}
	else
	{
		f += "0";
	}
	if (frq > 99)
	{
		val = frq / 100;
		f += String(val);
		frq -= val * 100;
	}
	else
	{
		f += "0";
	}
	if (frq > 9)
	{
		val = frq / 10;
		f += String(val);
		frq -= val * 10;
	}
	else
	{
		f += "0";
	}
	f += String(frq);
	return f;
}

String Flex_CalcMode()
{
	//calculates the correct flex mode from the ctr2 modes
	String md = "";
	switch (gRMode)
	{
	case RADIO_CW:
		md = "cw";
		break;
	case RADIO_CW_R:
		md = "cw";
		break;
	case RADIO_LSB:
		md = "lsb";
		break;
	case RADIO_USB:
		md = "usb";
		break;
	case RADIO_FM_N:
		md = "nfm"; //3khz bw
		break;
	case RADIO_FM_W:
		md = "fm"; //5khz bw
		break;
	case RADIO_AM:
		md = "am";
		break;
	case RADIO_DIG_L:
		md = "digl";
		break;
	case RADIO_DIG_H:
		md = "digu";
		break;
	}
	return md;
}


String Flex_ChkRx()
{
	char chr = 0;
	String txt = " "; //must have something in it
	String sliceListReply = "R2|0";
	static bool alreadyHr = false; //prevent recursion

	if (alreadyHr == true)
	{
		return txt;
	}
	alreadyHr = true;

	if (flexAPI.available())
	{
	TryAgn:
		chr = flexAPI.read();

#ifdef DEBUG_FLEX_API
		SerialOut(String(chr), false);
#endif

		//ignore cr & lf
		if (chr != 13 && chr != 10)
		{
			txt += String(chr);
		}
		else
		{
			chr = ' '; //force it to space
			txt += " "; //replace cr or lf with space so parser works
		}

		if (txt.length() > 100)
		{
			txt = txt.substring(50, txt.length()); //trim size otherwise we lock up proc
		}
		//chk for spontaneous slice and tx subscription updates

		if (txt.length() > 4)
		{
			//we always start with a ping first then get the slice list			

			int ptr = txt.indexOf(sliceListReply);
			if (ptr > -1)
			{
				//get next two chrs			
				uint32_t waitTmr = millis();
				while (waitTmr == millis())
				{
					//wait 1 mS
				}
				if (flexAPI.available() > 1)
				{
					chr = flexAPI.read();
					txt += String(chr);
					chr = flexAPI.read();
					txt += String(chr);
				}
				SerialOut(txt, true);
				sliceListReply += "|0"; //chk for 0 slice								
				if (txt.indexOf(sliceListReply) == -1)
				{
					//slice A doesn't exist, create it				
					String frq = Flex_CalcFreq(gFreqRadio);
					String md = Flex_CalcMode();
					txt = "c" + String(gFlexCmdNum) + "|slice create freq=" + frq + " ant=ANT" + String(gFlexAnt) + " mode=" + md + "\n\r";
				}
				alreadyHr = false;
				return txt;
			}

			//chk for slice subscription replies - they are "=" pairs separated by a space
			if (chr == ' '  && txt.indexOf("=") > -1)
			{
				Flex_Parse_Rx(txt);
			}
		}
		uint32_t waitTmr = millis();
		while (waitTmr == millis())
		{
			//wait 1 mS
		}
		if (flexAPI.available())
		{
			goto TryAgn;
		}
		gFlexLastReply = millis(); //UpdateHomePage() uses this to determine if radio is online
	}
	alreadyHr = false;
	return txt;

}

void Flex_Parse_Rx(String & txt)
{
	String data = ""; //string returned from Flex_Extract()
	int val = 0;

	static bool slice0 = false; //goes true if last slice update was for slice 0. goes false if any other slice update preceeded this pair

	//parse txt for '=' pair - if found, update variable associated with it. Clear txt if '=' pair found
	if (txt.length() == 0)
	{
		txt = " "; //make sure there's something in it or we might lock up proc
	}

	//set the static slice0 value - true when last |slice frame is for slice 0
	if (txt.indexOf("|slice ") > -1)
	{
		if (txt.indexOf("|slice 0") > -1)
		{
			slice0 = true; //accept slice update replies
		}
		else
		{
			slice0 = false; //ignore any slice replies
		}
	}

	if (slice0 == true)
	{
		//look for slice updates
		data = Flex_Extract(" mode=", txt);
		if (data.length() > 0)
		{
			if (data == "CW")
			{
				gRMode = RADIO_CW;
			}
			else if (data == "LSB")
			{
				gRMode = RADIO_LSB;
			}
			else if (data == "USB")
			{
				gRMode = RADIO_USB;
			}
			else if (data == "AM")
			{
				gRMode = RADIO_AM;
			}
			else if (data == "NFM")
			{
				gRMode = RADIO_FM_N;
			}
			else if (data == "FM")
			{
				gRMode = RADIO_FM_W;
			}
			else if (data == "DIGL")
			{
				gRMode = RADIO_DIG_L;
			}
			else if (data == "DIGU")
			{
				gRMode = RADIO_DIG_H;
			}
			hmiWait4Reply = 0;
			Tx2HMI("gRMode=" + String(gRMode));
			Tx2HMI("gRMode=" + String(gRMode));
			//gRModePrev = gRMode; //done by Dec_Home_Data
			return;
		}
		data = Flex_Extract("wnb=", txt);
		if (data.length() > 0)
		{
			//update noise blanker status
			SerialOut("==> wnb= " + data, true);
			val = data.toInt();
			gFlexWNB = (gFlexWNB & 0x7f) + (val << 7);
			gFlexWNBPrev = gFlexWNB;
			hmiWait4Reply = 0;
			Tx2HMI("Flex.gFlexWNB.val=" + String(gFlexWNB));
			Tx2HMI("Flex.gFlexWNB.val=" + String(gFlexWNB));
			return;
		}
		data = Flex_Extract(" nb=", txt);
		if (data.length() > 0)
		{
			//update noise blanker status
			SerialOut("==> nb= " + data, true);
			val = data.toInt();
			gFlexNB = (gFlexNB & 0x7f) + (val << 7);
			gFlexNBPrev = gFlexNB;
			hmiWait4Reply = 0;
			Tx2HMI("Flex.gFlexNB.val=" + String(gFlexNB));
			Tx2HMI("Flex.gFlexNB.val=" + String(gFlexNB));
			return;
		}
		data = Flex_Extract(" nr=", txt);
		if (data.length() > 0)
		{
			//update noise blanker status
			SerialOut("==> nr= " + data, true);
			val = data.toInt();
			gFlexNR = (gFlexNR & 0x7f) + (val << 7);
			gFlexNRPrev = gFlexNR;
			hmiWait4Reply = 0;
			Tx2HMI("Flex.gFlexNR.val=" + String(gFlexNR));
			Tx2HMI("Flex.gFlexNR.val=" + String(gFlexNR));
			return;
		}
		if (gRMode < 2)
		{
			//use auto peaking filter for cw modes
			data = Flex_Extract("apf=", txt);
			if (data.length() > 0)
			{
				SerialOut("==> apf= " + data, true);
				val = data.toInt();
				gFlexAF = (gFlexAF & 0x7f) + (val << 7);
				gFlexAFPrev = gFlexAF;
				hmiWait4Reply = 0;
				Tx2HMI("Flex.gFlexAF.val=" + String(gFlexAF));
				Tx2HMI("Flex.gFlexAF.val=" + String(gFlexAF));
				return;
			}
		}
		else
		{
			//use auto notch filter for all other modes
			data = Flex_Extract("anf=", txt);
			if (data.length() > 0)
			{
				//update noise blanker status
				SerialOut("==> anf= " + data, true);
				val = data.toInt();
				gFlexAF = (gFlexAF & 0x7f) + (val << 7);
				gFlexAFPrev = gFlexAF;
				hmiWait4Reply = 0;
				Tx2HMI("Flex.gFlexAF.val=" + String(gFlexAF));
				Tx2HMI("Flex.gFlexAF.val=" + String(gFlexAF));
				return;
			}
		}
		data = Flex_Extract("wnb_level=", txt);
		if (data.length() > 0)
		{
			//update noise blanker status
			SerialOut("==> wnb_level= " + data, true);
			gFlexWNB = (gFlexWNB & 0x80) + data.toInt();
			gFlexWNBPrev = gFlexWNB;
			hmiWait4Reply = 0;
			Tx2HMI("Flex.gFlexWNB.val=" + String(gFlexWNB));
			Tx2HMI("Flex.gFlexWNB.val=" + String(gFlexWNB));
			SerialOut("> > > > gFlexWNB= " + String(gFlexWNB), true);
			return;
		}
		data = Flex_Extract(" nb_level=", txt);
		if (data.length() > 0)
		{
			//update noise blanker status
			SerialOut("==> nb= " + data, true);
			gFlexNB = (gFlexNB & 0x80) + data.toInt();
			gFlexNBPrev = gFlexNB;
			hmiWait4Reply = 0;
			Tx2HMI("Flex.gFlexNB.val=" + String(gFlexNB));
			Tx2HMI("Flex.gFlexNB.val=" + String(gFlexNB));
			return;
		}
		data = Flex_Extract(" nr_level=", txt);
		if (data.length() > 0)
		{
			//update noise blanker status
			SerialOut("==> nr= " + data, true);
			gFlexNR = (gFlexNR & 0x80) + data.toInt();
			gFlexNRPrev = gFlexNR;
			hmiWait4Reply = 0;
			Tx2HMI("Flex.gFlexNR.val=" + String(gFlexNR));
			Tx2HMI("Flex.gFlexNR.val=" + String(gFlexNR));
			return;
		}
		if (gRMode < 2)
		{
			//use auto peaking filter for CW
			data = Flex_Extract("apf_level=", txt);
			if (data.length() > 0)
			{
				//update auto peaking filter status				
				gFlexAF = (gFlexAF & 0x80) + data.toInt();
				gFlexAFPrev = gFlexAF;
				hmiWait4Reply = 0;
				Tx2HMI("Flex.gFlexAF.val=" + String(gFlexAF));
				Tx2HMI("Flex.gFlexAF.val=" + String(gFlexAF));
				SerialOut("....apf -- gFlexAF= " + String(gFlexAF), true);
				return;
			}
		}
		else
		{
			//use auto notch for other modes
			data = Flex_Extract("anf_level=", txt);
			if (data.length() > 0)
			{
				//update auto notch filter status				
				gFlexAF = (gFlexAF & 0x80) + data.toInt();
				gFlexAFPrev = gFlexAF;
				hmiWait4Reply = 0;
				Tx2HMI("Flex.gFlexAF.val=" + String(gFlexAF));
				Tx2HMI("Flex.gFlexAF.val=" + String(gFlexAF));
				SerialOut("....anf -- gFlexAF= " + String(gFlexAF), true);
				return;
			}
		}
	}

	//misc info
	data = Flex_Extract("model=", txt);
	if (data.length() > 0)
	{
		SerialOut("===>model= " + data, true);
		return;
	}
	data = Flex_Extract("nickname=", txt);
	if (data.length() > 0)
	{
		SerialOut("===>nickname= " + data, true);
		txt = "";
		return;
	}
	data = Flex_Extract("callsign=", txt);
	if (data.length() > 0)
	{
		SerialOut("===>callsign= " + data, true);
		return;
	}

	//rx info
	data = Flex_Extract("RF_frequency=", txt);
	if (data.length() > 0)
	{
		SerialOut("==> decode frq= " + data, true);
		int32_t chk = int32_t(data.toFloat() * 1000000);
		if (chk >= 100000)
		{
			gFreqRadio = chk;
			ChkFreqChange();
			//SerialOut("set rx freq= " + String(gFreqRadio), true);
		}
		return;
	}
	data = Flex_Extract("lineout_gain=", txt);
	if (data.length() > 0)
	{
		//update rx line out
		SerialOut("==> lineout= " + data, true);
		gFlexLO = data.toInt();
		gFlexLOPrev = gFlexLO;
		hmiWait4Reply = 0;
		Tx2HMI("Flex.gFlexLO.val=" + String(gFlexLO));
		Tx2HMI("Flex.gFlexLO.val=" + String(gFlexLO));
		return;
	}

	//tx info
	data = Flex_Extract("rfpower=", txt);
	if (data.length() > 0)
	{
		//update tx power
		SerialOut("==> tx power= " + data, false);
		gFlexPO = data.toInt();
		gFlexPOPrev = gFlexPO; //so we don't send it back in UpdateHMIDisplay()		
		Tx2HMI("Flex.gFlexPO.val=" + String(gFlexPO));
		Tx2HMI("Flex.gFlexPO.val=" + String(gFlexPO));
		return;
	}

	data = Flex_Extract("tunepower=", txt);
	if (data.length() > 0)
	{
		//update tx tune power
		SerialOut("==> tx tune power= " + data, true);
		gFlexTO = data.toInt();
		gFlexTOPrev = gFlexTO; //so we don't send it back in UpdateHMIDisplay()		
		Tx2HMI("Flex.gFlexTO.val=" + String(gFlexTO));
		Tx2HMI("Flex.gFlexTO.val=" + String(gFlexTO));
		return;
	}

	//atu info
	data = Flex_Extract("atu_enabled=", txt);
	if (data.length() > 0)
	{
		//update atu btn status
		SerialOut("==> atu enabled=(" + data + ")", true);
		if (data == "0")
		{
			//no atu - hide buttons
			Tx2HMI("Flex.gFlexATU.val=255");
			Tx2HMI("Flex.gFlexATU.val=255");
		}
		return;
	}
	data = Flex_Extract("status=", txt);
	if (data == "TUNE_MANUAL_BYPASS")
	{
		SerialOut("==> atu state=" + data, true);
		Tx2HMI("Flex.gFlexATU.val=0");
		Tx2HMI("Flex.gFlexATU.val=0");
		return;
	}
	else if (data == "TUNE_IN_PROGRESS")
	{
		SerialOut("==> atu state=" + data, true);
		Tx2HMI("Flex.gFlexATU.val=254");
		Tx2HMI("Flex.gFlexATU.val=254");
		return;
	}
	else if (data == "TUNE_SUCCESSFUL")
	{
		SerialOut("==> atu state=" + data, true);
		Tx2HMI("Flex.gFlexATU.val=1");
		Tx2HMI("Flex.gFlexATU.val=1");
		return;
	}

}


String Flex_Extract(String match, String & txt)
{
	//look for match in txt, if found, extract data value and return with txt empty, otherwise just return	
	String rtnData = "";
	int ptr = txt.indexOf(match);
	if (ptr > -1 && (ptr + match.length()) < txt.length())
	{
		//get data between '=' and ' ' or cr	
		rtnData = txt.substring(ptr + match.length(), txt.length()); // txt.indexOf(" ", ptr + match.length() + 1));
		SerialOut("....rtnData= " + rtnData, true);
		txt = " "; //leave something in txt
	}
	return rtnData.trim();
}


void Flex_TxFreq(uint32_t freq)
{
	//we always use slice A
	if (gWiFi == 1)
	{
		if (!flexAPI)
		{
			Flex_Initialize();
		}
		//we're online - send freq command	
		String frq = Flex_CalcFreq(gFreqRadio);
		Tx2ESP(ESP_FLEX, "|slice t 0 " + frq);
	}
}


void Flex_TxMode()
{
	//we always use slice A
	if (!flexAPI)
	{
		Flex_Initialize();
	}
	String md = Flex_CalcMode();
	Tx2ESP(ESP_FLEX, "|slice s 0 mode=" + md + " tx=1 txant=ANT" + String(gFlexAnt));
}
