%RFpwr.m
%
%With a single DC-biased detector diode (one 1N5711), RFpwr.m implements
%an accurate and inexpensive RF power meter for measuring small amounts of
%RF power delivered to Rin = 50 Ohms.
%
%See the paper:
% John Stensby, "A Diode-Model-Based RF Power Sensor", Under Review, QEX,
% American Radio Relay League, Newington, CT.
%
%The diode detector thermal voltage and the emission coefficient are denoted as
%Vt and N, respectively. To keep things simple, and without loss of generality,
%the product N(Vt)= NVt is called thermal voltage. You should find that NVt is
%APPROXIMATELY .0258 volts at room temp.
%
%Both NVt and the input power (absorbed by the input fixed Rin = 50 Ohms)
%are calculated and displayed on the computer monitor.

%Program inputs and outputs are described below.
%
% John Stensby, N5DF
% stensbj@uah.edu
%
%Last Update: Nov 4, 2023
%
%---------------------------Nov 4, 2023----------------------------------------
%In microamps, Id1 and Id2 are DC bias currents. Measure your current values
%and list them here. These currents must be measured accurately.
Id1 = 1.00; %Specified in uA;
Id2 = 1.78; %Specified in uA;

%Measurements made at Current Port female pin jacks mounted on the Power Sensor
%aluminum enclosure. To make current and voltage measurements, I use a SIX &
%ONE-HALF digit, Siglent SDM3065X desk-top DMM.

%For Id1 and Id2 measurement, I let the DMM continuously sample and compute
%a moving average and standard deviation after each sample. After averaging a
%few tens of samples, the steady state is reached where the computed standard
%deviation is no longer decreasing significantlly. At this point, in RFpwr.m,
%I use the steady-state current value.

%Display 16 digit, IEEE format floating point numbers
format long

%Rin = the fixed precision non-inductive resistor used as the PwrSens input
%termination
Rin = 50;

%Rs is the parallel combination of Rin = 50 Ohms and the DC resistance looking
%back into the RF source output connector. *ALL* diode bias current flows
%throuth Rs. Two common cases:
%(1)The RF source is AC coupled or a DC block is used. In this case Rs = Rin
%   = 50 Ohms.
%(2)The DC resistance looking back into the RF source connector is 50 Ohms.
%   In this case Rs = .5*Rin = 25 Ohms.
%Otherwise, Rs < Rin if neither of the above cases applies.

%Substitute your Rs value here, or choose default Rs = Rin.

Rs = Rin; %Since my Rigol DS815 RF generator has an AC coupled output.

%Rd is the diode series resistance in the diode Spice Model.
Rd = 10;

%The total DC drop = (Rs+Rd)*Id is small compared to the voltage across the
%diode junction.  Hence an error in the specification of Rs+Rd may not make a
%significant difference.

%Turn off the RF input before entering PwrSens output voltages V1 for Id1
%and V2 for Id2.

disp('Turn off input RF before entering voltage V1 for Id1 & V2 for Id2')
disp(' ')

%V1 (for Id1), V2 (for Id2) and V3 (for RF on) are measured at the Voltage Port
V1 = input('For Id1 Enter V1  = ');
V2 = input('For Id2 Enter V2  = ');

%Vd1 = Junction DC for RF Off case & Id = Id1
Vd1 = -V1 - (Rd+Rs)*Id1*1e-6;

%Vd2 = Junction DC for RF Off case & Id = Id2
Vd2 = -V2 - (Rd+Rs)*Id2*1e-6;

Ratio = Id2/Id1;

%Thermal voltage x = N*Vt is the root x of the following nonlinear eqn.
%f1 implements Equation 8 in the above-referenced paper.
f1 = @(x) Ratio*(exp(Vd1/x) - 1) - (exp(Vd2/x) - 1);

%Solve f1(x) = 0 for x = NVt.
%Close approximation for N*Vt is x0 = (Vd2-Vd1)/log(Ratio)
x0 = (Vd2-Vd1)/log(Ratio);
NVt = fzero(f1,x0);

%Output N*(Thermal Voltage) = N*Vt
formatSpec = 'Thermal Voltage = N*Vt = %8.6f\n';
fprintf(formatSpec,NVt)

disp(' ')
%Either terminate with CTRL C (press CTRL and C keys simultaneously) or
%enter a carriage return to calculate input RF power.
input('Enter CTRL C to terminate; Or Enter carriage return to continue. ');
disp(' ')

%Turn On RF Input and Measure Voltage Port Value V3.
disp('Set current back to Id1 AND turn on RF input')
fprintf(' \n')

V3 = input('For RF On Case V3 = ');
fprintf(' \n')

%Junction DC for RF ON case and Id = Id1
Vd3 = -V3 - (Rd+Rs)*Id1*1e-6;

%f2 implements Equation (14) in the paper.
f2 = @(x) besseli(0,x/NVt,1) - exp((Vd1-Vd3-x)/NVt);
%Vd1-Vd3 is the initial guess for the root
M = fzero(f2,Vd1-Vd3);

%Output M
formatSpec = 'Peak RF Volts Across Input 50 Ohm Load = %8.6f\n\n';
fprintf(formatSpec,M)

%Power in milliwatts
pwr = 10*M^2;

%Power in dBm
dBm = 10*log10(pwr);

%Output Power in milliwatts and dBm
formatSpec = 'Power = %8.6f Milliwats OR %8.6f dBm\n\n';
fprintf(formatSpec,pwr,dBm)

%--------------------------------------------------------------------------
%Input new V3 > 0 for another power calculation using Existing NVt
% OR Enter V3 = 0 to stop
%disp('Enter new V3 > 0 for another calculation OR V3 = 0 to stop ')

V3 = input('Enter new V3 > 0 OR V3 = 0 To Stop. Enter V3 = ');

while V3 ~= 0 %Calculate New pwr with Existing NVt

%Junction DC for RF ON case and Id = Id1
    Vd3 = -V3 - (Rd+Rs)*Id1*1e-6;
    f2 = @(x) besseli(0,x/NVt,1) - exp((Vd1-Vd3-x)/NVt);
    M = fzero(f2,Vd1-Vd3);

 %Output M
    formatSpec='Peak RF Volts Across Input Termination = %8.6f\n\n';
    fprintf(formatSpec,M)

 %Calculate Power in Milliwatts
    pwr = 10*M^2;
 %Output Pwr in dBm
    dBm = 10*log10(pwr);

 %Output Pwr in Milliwatts and dBm
    formatSpec = 'Power = %8.6f Milliwats OR %8.6f dBm\n\n';
    fprintf(formatSpec,pwr,dBm)

    V3 = input('Enter new V3 > 0 OR V3 = 0 to stop.  Enter V3 = ');
end
